#' Test whether ref() and alt() are single nucleotides
#'
#' This function takes a 'VCF' object and returns a logical vector
#' indicating whether variants are 'simple'. A simple variant is one
#' where (a) the REF allele is a single character; (b) the is only one
#' ALT allele; and (c) the ALT allele is a single character
#'
#' @param vcf input VCF object
#' @return logical vector of length(vcf) indicating whether
#'     variant is simple
#'
#' @importMethodsFrom VariantAnnotation ref alt
#' @importMethodsFrom Biostrings nchar
#' @importMethodsFrom IRanges unlist relist elementLengths
#' @export
#'
#' @examples
#' library(VariantAnnotation)
#' fl <- system.file(package="VariantAnnotation", "extdata",
#'    "chr22.vcf.gz")
#' vcf <- readVcf(fl, "hg19")
#' table(isSimpleVariant(vcf))
#' vcf[isSimpleVariant(vcf)]    # select simple variants only
isSimpleVariant <- function(vcf)
{
    stopifnot(is(vcf, "VCF"))
    refOK <- nchar(ref(vcf)) == 1
    alt <- alt(vcf)                     # DNAStringSet*List*
    isAlt1Char <- relist(nchar(unlist(alt)) == 1, alt)
    altOK <- (elementLengths(alt(vcf)) == 1) & (all(isAlt1Char))
    refOK & altOK
}
