test_that("Spectra,ANY works", {
    df <- DataFrame()
    res <- Spectra(df)
    expect_s4_class(res@backend, "MsBackendMemory")
    expect_true(validObject(res))
    expect_true(length(res) == 0)

    df <- DataFrame(msLevel = c(1L, 2L))
    res <- Spectra(df)
    expect_s4_class(res@backend, "MsBackendMemory")
    expect_identical(msLevel(res), c(1L, 2L))
    expect_true(length(res) == 2)

    df$polarity <- "NEG"
    expect_error(Spectra(df), "wrong data type: polarity")

    res <- Spectra(files = sciex_file, source = MsBackendMzR())
    expect_s4_class(res@backend, "MsBackendMzR")
    expect_true(length(res) > 1)
})

test_that("Spectra,missing works", {
    res <- Spectra()
    expect_true(length(res) == 0)
    expect_s4_class(res@backend, "MsBackendMemory")

    res <- Spectra(backend = MsBackendDataFrame())
    expect_true(length(res) == 0)
    expect_s4_class(res@backend, "MsBackendDataFrame")

    res <- Spectra(source = MsBackendDataFrame())
    expect_true(length(res) == 0)
    expect_s4_class(res@backend, "MsBackendDataFrame")

    be <- backendInitialize(MsBackendDataFrame(), DataFrame(msLevel = c(1L, 2L),
                                                            fromFile = 1L))
    res <- Spectra(backend = be)
    expect_s4_class(res@backend, "MsBackendDataFrame")
    expect_true(length(res) == 2)
    expect_identical(msLevel(res), c(1L, 2L))
})

test_that("Spectra,MsBackend works", {
    be <- backendInitialize(MsBackendDataFrame(),
                            DataFrame(msLevel = c(1L, 2L),
                                      fromFile = 1L))
    res <- Spectra(be)
    expect_true(length(res) == 2)
    expect_identical(msLevel(res), c(1L, 2L))
})

test_that("Spectra,character works", {
    res <- Spectra(sciex_file)
    expect_true(is(res@backend, "MsBackendMzR"))
    expect_true(length(res) > 0)

    res <- Spectra(sciex_file, source = MsBackendMzR())
    expect_true(is(res@backend, "MsBackendMzR"))
    expect_equal(unique(res@backend$dataStorage), sciex_file)
    expect_identical(rtime(res), rtime(sciex_mzr))

    res_2 <- Spectra(sciex_file, source = MsBackendMzR(),
                     backend = MsBackendDataFrame())
    expect_true(is(res_2@backend, "MsBackendDataFrame"))
    expect_identical(rtime(res), rtime(res_2))

    show(res)

    ## Empty character
    res <- Spectra(character())
    expect_s4_class(res, "Spectra")
    expect_s4_class(res@backend, "MsBackendMzR")
    expect_true(length(res) == 0)

    res <- Spectra(character(), backend = MsBackendDataFrame())
    expect_s4_class(res, "Spectra")
    expect_s4_class(res@backend, "MsBackendDataFrame")
    expect_true(length(res) == 0)
})

test_that(".create_spectra works, ", {
    ## missing object
    res <- .create_spectra()
    expect_true(length(res) == 0)
    expect_s4_class(res@backend, "MsBackendMemory")
    expect_error(res <- .create_spectra(backend = MsBackendMzR()), "mandatory")

    ## object being a character, backend a MsBackendMemory -> error
    res <- expect_error(.create_spectra(sciex_file), "DataFrame")
    ## object being a character, backend a MsBackendMzR
    res <- .create_spectra(sciex_file, backend = MsBackendMzR())
    expect_s4_class(res@backend, "MsBackendMzR")
    dta <- spectraData(res@backend)

    ## object being a DataFrame, backend a MsBackendDataFrame
    res <- .create_spectra(dta, backend = MsBackendDataFrame())
    expect_s4_class(res@backend, "MsBackendDataFrame")
    expect_equal(res$msLevel, dta$msLevel)

    ## object missing but providing files
    res <- .create_spectra(files = sciex_file, backend = MsBackendMzR())
    expect_s4_class(res@backend, "MsBackendMzR")
    expect_equal(res$msLevel, dta$msLevel)

    ## object missing but providing data
    res <- .create_spectra(data = dta, backend = MsBackendMemory())
    expect_s4_class(res@backend, "MsBackendMemory")
    expect_equal(res$msLevel, dta$msLevel)

})

test_that("setBackend,Spectra works", {
    df <- DataFrame(rtime = as.numeric(1:9),
                    fact = c(2L, 1L, 2L, 1L, 3L, 2L, 3L, 3L, 1L))
    sps <- Spectra(df)
    res <- setBackend(sps, MsBackendDataFrame())
    expect_true(ncol(sps@backend@spectraData) < ncol(res@backend@spectraData))
    expect_identical(sps@backend@spectraData$fact,
                     res@backend@spectraData$fact)
    expect_identical(rtime(res), rtime(sps))
    expect_identical(dataStorage(res), dataStorage(sps))
    expect_identical(dataOrigin(res), dataOrigin(sps))

    ## Empty backends
    e <- sps[integer()]
    e2 <- setBackend(e, MsBackendDataFrame())
    expect_true(length(e2) == 0)
    expect_s4_class(e2@backend, "MsBackendDataFrame")
    expect_equal(e2$mz, IRanges::NumericList(compress = FALSE))
    e3 <- setBackend(e, MsBackendMemory())
    expect_true(length(e3) == 0)
    expect_s4_class(e3@backend, "MsBackendMemory")
    expect_equal(e3$mz, IRanges::NumericList(compress = FALSE))

    ## Use a different factor.
    res <- setBackend(sps, MsBackendDataFrame(), f = df$fact)
    expect_true(ncol(sps@backend@spectraData) < ncol(res@backend@spectraData))
    expect_identical(as.vector(sps@backend@spectraData$fact),
                     as.vector(res@backend@spectraData$fact))
    expect_identical(dataStorage(res), dataStorage(sps))
    expect_identical(rtime(res), rtime(sps))

    ## switch from mzR to DataFrame
    sps <- Spectra(sciex_mzr)
    res <- setBackend(sps, MsBackendDataFrame())
    expect_identical(rtime(sps), rtime(res))
    expect_identical(mz(sps), mz(res))
    expect_true(is(res@backend@spectraData$msLevel, "integer"))
    expect_true(is(sps@backend@spectraData$msLevel, "integer"))
    expect_true(is.integer(res$msLevel))
    expect_identical(dataOrigin(res), dataStorage(sps))

    ## switch from DataFrame to hdf5
    tdir <- normalizePath(paste0(tempdir(), "/a"))
    res <- setBackend(sps, MsBackendHdf5Peaks(), hdf5path = tdir)
    expect_identical(rtime(sps), rtime(res))
    expect_identical(peaksData(sps), peaksData(res))
    expect_identical(dataOrigin(res), dataStorage(sps))

    ## from DataFrame to hdf5 providing file names - need to disable
    ## parallelization
    res <- setBackend(sps, MsBackendHdf5Peaks(),
                      files = c(tempfile(), tempfile()),
                      f = rep(1, length(sps)))
    expect_identical(rtime(sps), rtime(res))
    expect_identical(peaksData(sps), peaksData(res))

    ## errors:
    expect_error(setBackend(sps, MsBackendMzR()), "support")
    expect_error(setBackend(sps, MsBackendMzR()), "MsBackendMzR")
})

test_that("c,Spectra works", {
    df1 <- DataFrame(msLevel = c(1L, 1L, 1L))
    df1$mz <- list(c(1.1, 1.2), c(1.5), c(1.4, 1.5, 1.6))
    df1$intensity <- list(c(4.5, 23), 452.1, c(4.1, 342, 123))
    sp1 <- Spectra(df1)

    df2 <- DataFrame(msLevel = c(2L, 2L), rtime = c(1.2, 1.5))
    df2$mz <- list(1.5, 1.5)
    df2$intensity <- list(1234.1, 34.23)
    sp2 <- Spectra(df2)

    df3 <- DataFrame(msLevel = c(3L, 3L), other_col = "a")
    df3$mz <- list(c(1.4, 1.5, 1.6), c(1.8, 1.9))
    df3$intensity <- list(c(123.4, 12, 5), c(43.1, 5))
    sp3 <- Spectra(df3)

    df4 <- df3
    df4$mz <- NULL
    df4$intensity <- NULL
    sp4 <- Spectra(df4)

    res <- c(sp1, sp2, sp3)
    expect_true(is(res, "Spectra"))
    expect_equal(length(res), sum(nrow(df1), nrow(df2), nrow(df3)))
    expect_identical(msLevel(res), c(1L, 1L, 1L, 2L, 2L, 3L, 3L))
    expect_identical(res$other_col, c(NA, NA, NA, NA, NA, "a", "a"))
    expect_true(length(res@processingQueue) == 0)
    expect_true(length(res@processing) == 1)

    res <- c(a = sp1, b = sp2, sp3)
    expect_true(is(res, "Spectra"))

    tmp <- vector("list", 3)
    tmp[[1L]] <- sp1
    tmp[[2L]] <- sp2
    tmp[[3L]] <- sp3

    res <- concatenateSpectra(tmp)
    expect_true(is(res, "Spectra"))
    res_c <- do.call(c, tmp)
    res_c@processing <- ""
    res@processing <- ""
    expect_equal(res, res_c)

    names(tmp) <- c("a", "b", "c")
    res <- concatenateSpectra(tmp)
    expect_true(is(res, "Spectra"))
})

test_that("acquisitionNum,Spectra works", {
    sps <- Spectra()
    res <- acquisitionNum(sps)
    expect_identical(res, integer())
    df <- DataFrame(msLevel = c(1L, 1L))
    sps <- Spectra(df)
    res <- acquisitionNum(sps)
    expect_identical(res, c(NA_integer_, NA_integer_))
    df$acquisitionNum <- 1:2
    sps <- Spectra(df)
    res <- acquisitionNum(sps)
    expect_identical(res, 1:2)
})

test_that("centroided,centroided<-,Spectra works", {
    sps <- Spectra()
    res <- centroided(sps)
    expect_identical(res, logical())
    df <- DataFrame(msLevel = c(1L, 2L, 1L), centroided = c(TRUE, FALSE, TRUE))
    sps <- Spectra(df)
    res <- centroided(sps)
    expect_identical(res, c(TRUE, FALSE, TRUE))
    centroided(sps) <- c(FALSE, TRUE, FALSE)
    res <- centroided(sps)
    expect_identical(res, c(FALSE, TRUE, FALSE))
    centroided(sps) <- FALSE
    expect_true(all(centroided(sps) == FALSE))
    expect_error(centroided(sps) <- 3, "'logical' of length")
    expect_error(centroided(sps) <- c(TRUE, FALSE, FALSE, TRUE), "length 1 or 3")
})

test_that("collisionEnergy,collisionEnergy<-,Spectra works", {
    sps <- Spectra()
    res <- collisionEnergy(sps)
    expect_identical(res, numeric())

    df <- DataFrame(msLevel = c(1L, 2L, 2L))
    sps <- Spectra(df)
    res <- collisionEnergy(sps)
    expect_true(all(is.na(res)))
    expect_equal(length(res), length(sps))

    collisionEnergy(sps) <- c(1.2, 1.4, 1.7)
    res <- collisionEnergy(sps)
    expect_identical(res, c(1.2, 1.4, 1.7))

    df$collisionEnergy <- c(3.4, 4.3, 2.3)
    sps <- Spectra(df)
    res <- collisionEnergy(sps)
    expect_identical(res, c(3.4, 4.3, 2.3))

    expect_error(collisionEnergy(sps) <- 4, "of length 3")
    expect_error(collisionEnergy(sps) <- c("a", "b", "c"), "'numeric'")
})

test_that("dataOrigin,Spectra works", {
    sps <- Spectra()
    res <- dataOrigin(sps)
    expect_identical(res, character())

    df <- DataFrame(msLevel = c(1L, 2L))
    sps <- Spectra(df)
    res <- dataOrigin(sps)
    expect_identical(res, rep(NA_character_, length(sps)))

    df <- DataFrame(msLevel = c(1L, 2L), dataOrigin = c("a", "b"))
    sps <- Spectra(df)
    res <- dataOrigin(sps)
    expect_identical(res, c("a", "b"))

    sps <- Spectra(backend = sciex_mzr)
    res <- dataOrigin(sps)
    expect_identical(res, dataStorage(sps))
})

test_that("dataStorage,Spectra works", {
    sps <- Spectra()
    res <- dataStorage(sps)
    expect_identical(res, character())

    df <- DataFrame(msLevel = c(1L, 2L))
    sps <- Spectra(df)
    res <- dataStorage(sps)
    expect_identical(res, rep("<memory>", 2))

    sps <- Spectra(sciex_mzr)
    res <- dataStorage(sps)
    expect_identical(res, rep(sciex_file, each = 931))
})

test_that("length,Spectra works", {
    sps <- Spectra()
    expect_equal(length(sps), 0)

    df <- DataFrame(msLevel = c(1L, 2L))
    sps <- Spectra(df)
    expect_true(length(sps) == 2)
})

test_that("intensity,Spectra works", {
    sps <- Spectra()
    res <- intensity(sps)
    expect_true(is(res, "NumericList"))
    expect_true(length(res) == 0)

    df <- DataFrame(msLevel = c(1L, 1L),
                    rtime = c(1.2, 1.4),
                    centroided = TRUE)
    df$intensity <- list(c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                         c(9, 6, 0, 0, 3, 9, 4, 0, 0, 0))
    df$mz <- list(c(1:10), c(1:10))
    sps <- Spectra(df)
    res <- intensity(sps)
    expect_identical(res, NumericList(
                          c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                          c(9, 6, 0, 0, 3, 9, 4, 0, 0, 0),
                          compress = FALSE))
    sps <- filterIntensity(sps, intensity = 0.1)
    res <- intensity(sps)
    expect_identical(res, NumericList(c(1, 6, 3, 9, 1), c(9, 6, 3, 9, 4),
                                      compress = FALSE))
})

test_that("ionCount,Spectra works", {
    sps <- Spectra()
    res <- ionCount(sps)
    expect_identical(res, numeric())

    df <- DataFrame(msLevel = c(1L, 2L), centroided = TRUE)
    df$intensity <- list(c(5, 9, 3), c(9, 8, 2))
    df$mz <- list(1:3, 1:3)
    sps <- Spectra(df)

    res <- ionCount(sps)
    expect_identical(res, c(17, 19))

    sps <- replaceIntensitiesBelow(sps, threshold = 4)
    res <- ionCount(sps)
    expect_identical(res, c(14, 17))
})

test_that("isCentroided,Spectra works", {
    sps <- Spectra()
    res <- isCentroided(sps)
    expect_identical(res, logical())

    df <- DataFrame(msLevel = c(1L, 1L))
    df$intensity <- list(c(5, 6, 1), c(5, 3, 1))
    df$mz <- list(1:3, 1:3)
    sps <- Spectra(df)

    res <- isCentroided(sps)
    expect_identical(res, c(NA, NA))

    sps <- Spectra(sciex_mzr)
    res <- isCentroided(sps)
    expect_true(length(res) == length(sps))
    expect_true(all(!res))
})

test_that("isEmpty,Spectra works", {
    sps <- Spectra()
    res <- isEmpty(sps)
    expect_identical(res, logical())

    df <- DataFrame(msLevel = c(2L, 2L), centroided = TRUE)
    df$intensity <- list(c(4, 6, 1), c(45, 2))
    df$mz <- list(1:3, 1:2)

    sps <- Spectra(df)
    res <- isEmpty(sps)
    expect_identical(res, c(FALSE, FALSE))

    sps <- replaceIntensitiesBelow(sps, threshold = 100)
    res <- isEmpty(sps)
    expect_identical(res, c(FALSE, FALSE))

    sps <- filterIntensity(sps, intensity = 0.1)
    res <- isEmpty(sps)
    expect_identical(res, c(TRUE, TRUE))
})

test_that("isolationWindowLowerMz,Spectra works", {
    sps <- Spectra()
    expect_identical(isolationWindowLowerMz(sps), numeric())

    sps <- Spectra(sciex_mzr)
    expect_true(all(is.na(isolationWindowLowerMz(sps))))
    isolationWindowLowerMz(sps) <- as.numeric(1:length(sps))
    expect_identical(isolationWindowLowerMz(sps), as.numeric(1:length(sps)))

    sps <- Spectra(tmt_mzr)
    expect_true(all(is.na(isolationWindowLowerMz(sps)[msLevel(sps) == 1L])))
    expect_true(all(!is.na(isolationWindowLowerMz(sps)[msLevel(sps) == 2L])))
})

test_that("isolationWindowTargetMz,Spectra works", {
    sps <- Spectra()
    expect_identical(isolationWindowTargetMz(sps), numeric())

    sps <- Spectra(sciex_mzr)
    expect_true(all(is.na(isolationWindowTargetMz(sps))))
    isolationWindowTargetMz(sps) <- as.numeric(1:length(sps))
    expect_identical(isolationWindowTargetMz(sps), as.numeric(1:length(sps)))

    sps <- Spectra(tmt_mzr)
    expect_true(all(is.na(isolationWindowTargetMz(sps)[msLevel(sps) == 1L])))
    expect_true(all(!is.na(isolationWindowTargetMz(sps)[msLevel(sps) == 2L])))
    expect_true(all(isolationWindowTargetMz(sps)[msLevel(sps) == 2L] >
                    isolationWindowLowerMz(sps)[msLevel(sps) == 2L]))
})

test_that("isolationWindowUpperMz,Spectra works", {
    sps <- Spectra()
    expect_identical(isolationWindowUpperMz(sps), numeric())

    sps <- Spectra(sciex_mzr)
    expect_true(all(is.na(isolationWindowUpperMz(sps))))
    isolationWindowUpperMz(sps) <- as.numeric(1:length(sps))
    expect_identical(isolationWindowUpperMz(sps), as.numeric(1:length(sps)))

    sps <- Spectra(tmt_mzr)
    expect_true(all(is.na(isolationWindowUpperMz(sps)[msLevel(sps) == 1L])))
    expect_true(all(!is.na(isolationWindowUpperMz(sps)[msLevel(sps) == 2L])))
    expect_true(all(isolationWindowUpperMz(sps)[msLevel(sps) == 2L] >
                    isolationWindowTargetMz(sps)[msLevel(sps) == 2L]))
})

test_that("msLevel,Spectra works", {
    sps <- Spectra()
    expect_identical(msLevel(sps), integer())

    df <- DataFrame(msLevel = c(1L, 2L))
    sps <- Spectra(df)
    expect_identical(msLevel(sps), c(1L, 2L))
})

test_that("mz,Spectra works", {
    sps <- Spectra()
    res <- mz(sps)
    expect_true(is(res, "NumericList"))
    expect_true(length(res) == 0)

    df <- DataFrame(msLevel = c(1L, 1L),
                    rtime = c(1.2, 1.4),
                    centroided = TRUE)
    df$intensity <- list(c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                         c(9, 6, 0, 0, 3, 9, 4, 0, 0, 0))
    df$mz <- list(c(1:10), c(1:10))
    sps <- Spectra(df)
    res <- mz(sps)
    expect_identical(res, NumericList(1:10, 1:10, compress = FALSE))
    sps <- filterIntensity(sps, intensity = 0.1)
    res <- mz(sps)
    expect_equal(res, NumericList(c(3, 4, 5, 8, 9), c(1, 2, 5, 6, 7),
                                  compress = FALSE))
})

test_that("peaksData,Spectra works", {
    df <- DataFrame(msLevel = c(1L, 2L), fromFile = 1L)
    df$mz <- list(1:4, 1:5)
    df$intensity <- list(2:5, 1:5)
    be <- backendInitialize(MsBackendDataFrame(), file = NA_character_, df)
    sps <- Spectra(backend = be)
    res <- peaksData(sps)
    expect_true(is(res, "SimpleList"))
    expect_equal(res[[1]][, 1], 1:4)
    expect_equal(res[[2]][, 1], 1:5)
    res_2 <- as(sps, "SimpleList")
    expect_equal(res, res_2)
    res <- as(sps, "list")
    expect_true(is.list(res))
    expect_equal(res[[1]][, 1], 1:4)
    expect_equal(res[[2]][, 1], 1:5)

    sps <- Spectra(backend = MsBackendDataFrame())
    res <- peaksData(sps)
    expect_true(is(res, "SimpleList"))
    expect_true(length(res) == 0)

    res_2 <- as(sps, "SimpleList")
    expect_equal(res, res_2)
    res <- as(sps, "list")
    expect_true(is(res, "list"))
    expect_true(length(res) == 0)

    ## With columns
    ## MsBackendMzR
    sps <- Spectra(sciex_mzr)
    expect_error(peaksData(sps, columns = c("mz", "intensity", "other")),
                 "only support")
    res <- peaksData(sps, columns = c("intensity", "mz", "intensity"))
    expect_equal(res[[1L]][, 1L], res[[1L]][, 3L])
    expect_equal(res[[1L]][, 1L], intensity(sps)[[1L]])

    ## MsBackendDataFrame
    sps <- Spectra(be)
    expect_error(peaksData(sps, columns = c("mz", "intensity", "other")),
                 "not available")
    res <- peaksData(sps, columns = c("intensity", "mz", "intensity"))
    expect_equal(res[[1L]][, 1L], res[[1L]][, 3L])
    expect_equal(res[[1L]][, 1L], intensity(sps)[[1L]])

    ## MsBackendHdf5peaks
    sps <- Spectra(sciex_hd5)
    expect_error(peaksData(sps, columns = c("mz", "intensity", "other")),
                 "only support")
    res <- peaksData(sps, columns = c("intensity", "mz", "intensity"))
    expect_equal(res[[1L]][, 1L], res[[1L]][, 3L])
    expect_equal(res[[1L]][, 1L], intensity(sps)[[1L]])

    ## MsBackendMemory with additional peaks variables.
    tmp <- data.frame(rtime = c(1.1, 1.2, 1.3, 1.4),
                      msLevel = 1L)
    tmp$mz <- list(c(13, 14.1, 22, 23, 24, 49),
                   c(45.1, 56),
                   c(34.3, 134.4, 344, 443),
                   c(12.1, 31))
    tmp$intensity <- list(c(100, 300, 30, 120, 12, 34),
                          c(345, 234),
                          c(123, 124, 145, 3),
                          c(122, 421))
    tmp$ann <- list(c("a", NA, "b", "c", "d", NA),
                    c("e", "f"),
                    c("g", "h", "i", NA),
                    c("j", "k"))
    s <- Spectra(tmp, peaksVariables = c("mz", "intensity", "ann"))

    res <- peaksData(s, columns = "ann")
    expect_true(is.data.frame(res[[1L]]))
    res <- peaksData(s)
    expect_true(is.matrix(res[[1L]]))
    res <- peaksData(s, columns = peaksVariables(s))
    expect_true(is.data.frame(res[[1L]]))

    ## Filtering.
    s <- filterMzValues(s, 23, tolerance = 1)
    ref <- peaksData(s)
    expect_equal(ref[[1L]][, "mz"], c(22, 23, 24))

    res <- peaksData(s, columns = peaksVariables(s))
    expect_equal(res[[1L]][, "mz"], c(22, 23, 24))
    expect_true(is.data.frame(res[[1L]]))
    expect_equal(res[[1L]][, "ann"], c("b", "c", "d"))

    res <- peaksData(s, columns = c("ann"))
    expect_true(is.data.frame(res[[1L]]))
    expect_equal(res[[1L]][, "ann"], c("b", "c", "d"))
    expect_equal(colnames(res[[1L]]), "ann")

    res <- peaksData(s, columns = c("ann", "mz"))
    expect_true(is.data.frame(res[[1L]]))
    expect_equal(res[[1L]][, "ann"], c("b", "c", "d"))
    expect_equal(colnames(res[[1L]]), c("ann", "mz"))

    expect_equal(s$ann[[1L]], c("b", "c", "d"))

    expect_equal(spectraData(s, "ann")$ann[[1L]], c("b", "c", "d"))
    expect_equal(spectraData(s, c("rtime", "ann"))$ann[[1L]], c("b", "c", "d"))
    expect_equal(
        spectraData(s, c(spectraVariables(s), peaksVariables(s)))$ann[[1L]],
        c("b", "c", "d"))
})

test_that("lengths,Spectra works", {
    sps <- Spectra()
    res <- lengths(sps)
    expect_identical(res, integer())

    df <- DataFrame(msLevel = c(2L, 2L), centroided = TRUE)
    df$intensity <- list(c(4, 6, 1), c(45, 2))
    df$mz <- list(1:3, 1:2)

    sps <- Spectra(df)
    res <- lengths(sps)
    expect_identical(res, c(3L, 2L))

    sps <- replaceIntensitiesBelow(sps, threshold = 100)
    res <- lengths(sps)
    expect_identical(res, c(3L, 2L))

    sps <- filterIntensity(sps, intensity = 0.1)
    res <- lengths(sps)
    expect_identical(res, c(0L, 0L))
})

test_that("polarity,polarity<-,Spectra works", {
    sps <- Spectra()
    expect_identical(polarity(sps), integer())

    df <- DataFrame(msLevel = c(1L, 1L))
    sps <- Spectra(df)
    expect_identical(polarity(sps), c(NA_integer_, NA_integer_))
    polarity(sps) <- c(1L, -1L)
    expect_identical(polarity(sps), c(1L, -1L))
    polarity(sps) <- 2L
    expect_identical(polarity(sps), c(2L, 2L))

    expect_error(polarity(sps) <- c(1L, 2L, 1L), "of length 1 or 2")
    expect_error(polarity(sps) <- c("a", "b"), "an 'integer'")
})

test_that("precScanNum,Spectra works", {
    sps <- Spectra()
    expect_identical(precScanNum(sps), integer())

    sps <- Spectra(DataFrame(msLevel = c(2L, 2L)))
    expect_identical(precScanNum(sps), c(NA_integer_, NA_integer_))
    sps <- Spectra(DataFrame(msLevel = c(2L, 2L), precScanNum = c(1L, 2L)))
    expect_identical(precScanNum(sps), c(1L, 2L))
})

test_that("precursorCharge,Spectra works", {
    sps <- Spectra()
    expect_identical(precursorCharge(sps), integer())

    sps <- Spectra(DataFrame(msLevel = c(2L, 2L)))
    expect_identical(precursorCharge(sps), c(NA_integer_, NA_integer_))
    sps <- Spectra(DataFrame(msLevel = c(2L, 2L), precursorCharge = c(1L, 3L)))
    expect_identical(precursorCharge(sps), c(1L, 3L))
})

test_that("precursorIntensity,Spectra works", {
    sps <- Spectra()
    expect_identical(precursorIntensity(sps), numeric())

    sps <- Spectra(DataFrame(msLevel = c(2L, 2L)))
    expect_identical(precursorIntensity(sps), c(NA_real_, NA_real_))
    sps <- Spectra(DataFrame(msLevel = c(2L, 2L),
                             precursorIntensity = c(1.3, 3.2)))
    expect_identical(precursorIntensity(sps), c(1.3, 3.2))
})

test_that("precursorMz,Spectra works", {
    sps <- Spectra()
    expect_identical(precursorMz(sps), numeric())

    sps <- Spectra(DataFrame(msLevel = c(2L, 2L)))
    expect_identical(precursorMz(sps), c(NA_real_, NA_real_))
    sps <- Spectra(DataFrame(msLevel = c(2L, 2L),
                             precursorMz = c(234.2, 668.2)))
    expect_identical(precursorMz(sps), c(234.2, 668.2))
})

test_that("rtime,rtime<-,Spectra works", {
    sps <- Spectra()
    expect_identical(rtime(sps), numeric())

    sps <- Spectra(DataFrame(msLevel = c(1L, 2L)))
    expect_identical(rtime(sps), c(NA_real_, NA_real_))
    sps <- Spectra(DataFrame(msLevel = c(1L, 2L), rtime = c(2.1, 4.2)))
    expect_identical(rtime(sps), c(2.1, 4.2))
    rtime(sps) <- c(1.2, 1.3)
    expect_identical(rtime(sps), c(1.2, 1.3))

    expect_error(rtime(sps) <- c(TRUE, FALSE), "'numeric' of length 2")
    expect_error(rtime(sps) <- 1.3, "'numeric' of length 2")
})

test_that("scanIndex,Spectra works", {
    sps <- Spectra()
    expect_identical(scanIndex(sps), integer())
    sps <- Spectra(DataFrame(msLevel = c(1L, 2L)))
    expect_identical(scanIndex(sps), c(NA_integer_, NA_integer_))

    sps <- Spectra(DataFrame(msLevel = c(1L, 2L), scanIndex = c(1L, 2L)))
    expect_identical(scanIndex(sps), c(1L, 2L))
})

test_that("selectSpectraVariables,Spectra works", {
    sps <- Spectra()
    res <- selectSpectraVariables(sps, c("msLevel", "rtime"))
    expect_equal(sps, res)

    sps <- Spectra(DataFrame(msLevel = c(1L, 2L), rtime = c(1.2, 1.4),
                             other_col = "a"))
    res <- selectSpectraVariables(sps, c("rtime", "other_col"))
    expect_equal(msLevel(res), c(NA_integer_, NA_integer_))
    expect_equal(rtime(res), c(1.2, 1.4))
    expect_equal(spectraData(res, "other_col")[, 1], c("a", "a"))

    res <- selectSpectraVariables(sps, c("msLevel"))
    expect_identical(rtime(res), c(NA_real_, NA_real_))
    expect_identical(msLevel(res), 1:2)
    expect_true(!any(spectraVariables(res) %in% "other_col"))

    expect_error(selectSpectraVariables(sps, c("rtime", "something")), "not")
})

test_that("smoothed,smoothed<-,Spectra works", {
    sps <- Spectra()
    expect_identical(smoothed(sps), logical())

    sps <- Spectra(DataFrame(msLevel = 1:2))
    expect_identical(smoothed(sps), c(NA, NA))
    sps <- Spectra(DataFrame(msLevel = 1:2, smoothed = TRUE))
    expect_identical(smoothed(sps), c(TRUE, TRUE))

    smoothed(sps) <- c(FALSE, TRUE)
    expect_identical(smoothed(sps), c(FALSE, TRUE))
    smoothed(sps) <- FALSE
    expect_identical(smoothed(sps), c(FALSE, FALSE))

    expect_error(smoothed(sps) <- c("a", "b"), "of length 1 or 2")
    expect_error(smoothed(sps) <- c(TRUE, FALSE, TRUE), "of length 1 or 2")
})

test_that("spectraData,Spectra works", {
    sps <- Spectra()
    res <- spectraData(sps)
    expect_true(is(res, "DataFrame"))
    expect_true(nrow(res) == 0)
    expect_true(all(c("msLevel", "rtime") %in% colnames(res)))

    df <- DataFrame(msLevel = c(1L, 1L), rtime = c(1.2, 1.3), centroided = TRUE)
    df$mz <- list(1:10, 1:10)
    df$intensity <- list(c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                         c(9, 6, 0, 0, 3, 0, 0, 0, 3, 2))
    sps <- Spectra(df)
    res <- spectraData(sps)
    expect_true(is(res, "DataFrame"))
    expect_equal(res$msLevel, c(1L, 1L))
    expect_equal(res$rtime, c(1.2, 1.3))
    expect_equal(res$precScanNum, c(NA_integer_, NA_integer_))
    expect_equal(sps$mz, NumericList(1:10, 1:10, compress = FALSE))
    expect_equal(sps$intensity, NumericList(c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                                            c(9, 6, 0, 0, 3, 0, 0, 0, 3, 2),
                                            compress = FALSE))
    sps <- filterIntensity(sps, intensity = 0.1)
    res <- spectraData(sps)
    expect_equal(sps$mz, NumericList(c(3, 4, 5, 8, 9), c(1, 2, 5, 9, 10),
                                     compress = FALSE))
    expect_equal(sps$intensity, NumericList(c(1, 6, 3, 9, 1), c(9, 6, 3, 3, 2),
                                            compress = FALSE))
})

test_that("spectraData<-,Spectra works", {
    df <- DataFrame(msLevel = c(1L, 1L), rtime = c(1.2, 1.3), centroided = TRUE)
    df$mz <- list(1:10, 1:10)
    df$intensity <- list(c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                         c(9, 6, 0, 0, 3, 0, 0, 0, 3, 2))
    sps <- Spectra(df)

    spectraData(sps)$add_col <- 3
    expect_true(any(spectraVariables(sps) == "add_col"))
    expect_equal(spectraData(sps)$add_col, c(3, 3))

    expect_error(spectraData(sps) <- 4, "DataFrame")

    df$mz <- list(11:20, 11:20)
    df$precursorMz <- c(0, 0)

    spectraData(sps) <- df
    expect_true(!any(spectraVariables(sps) == "add_col"))
    expect_equal(mz(sps), NumericList(11:20, 11:20, compress = FALSE))

    expect_error(spectraData(sps) <- c(1, 2, 4, 5), "DataFrame")

    expect_error(spectraData(sps) <- df[c(1, 1, 2), ], "with 2 rows")

    sps_orig <- sps
    spectraData(sps)$rtime <- c(5, 2)
    expect_equal(sps$mz, sps_orig$mz)
    expect_equal(sps$intensity, sps_orig$intensity)
    expect_equal(sps$msLevel, sps_orig$msLevel)

    tmp <- sciex_mzr
    sps <- Spectra(tmp)
    sps <- sps[570:600]
    expect_warning(spectraData(sps)$some_col <- "yes", "replacing")
    expect_true(any(spectraVariables(sps) == "some_col"))
    expect_true(all(spectraData(sps, "some_col")[, 1] == "yes"))
    expect_true(is(sps@backend@spectraData$some_col, "character"))
    sps$other_col <- "other_value"
    expect_true(any(spectraVariables(sps) == "other_col"))
    expect_identical(sps$other_col, rep("other_value", length(sps)))

    sps_orig <- sps
    spd <- spectraData(sps)
    spd$msLevel <- 2L
    expect_warning(spectraData(sps) <- spd, "Ignoring")
    expect_true(all(msLevel(sps) == 2L))

    expect_equal(mz(sps), mz(sps_orig))
    expect_equal(intensity(sps), intensity(sps_orig))
})

test_that("spectraNames,spectraNames<-,Spectra works", {
    sps <- Spectra()
    expect_true(length(spectraNames(sps)) == 0)

    sps <- Spectra(DataFrame(msLevel = c(1L, 1L)))
    expect_identical(spectraNames(sps), as.character(1:2))

    spectraNames(sps) <- c("a", "b")
    expect_identical(spectraNames(sps), c("a", "b"))

    expect_error(spectraNames(sps) <- 1, "invalid")
})

test_that("spectraVariables,Spectra works", {
    sps <- Spectra()
    res <- spectraVariables(sps)
    exp_col <- c("msLevel", "rtime", "acquisitionNum", "scanIndex",
                 "dataStorage", "centroided", "smoothed",
                 "polarity", "precScanNum", "precursorMz", "precursorIntensity",
                 "precursorCharge", "collisionEnergy")
    expect_true(all(exp_col %in% res))
    df <- DataFrame(other_col = "a", msLevel = c(1L, 1L))
    sps <- Spectra(df)
    res <- spectraVariables(sps)
    expect_true(all(c(exp_col, "other_col") %in% res))

    df$mz <- list(c(1.2, 1.4), c(4.5, 5.6, 7.8))
    df$intensity <- list(c(12, 23.3), c(134.3, 5, 123))
    df$pk_ann <- list(c("a", "b"), c(NA, NA, NA))
    sps <- Spectra(df, peaksVariables = c("mz", "intensity", "pk_ann"))
    res <- spectraVariables(sps)
    expect_true(!any(peaksVariables(sps) %in% res))
})

test_that("tic,Spectra works", {
    sps <- Spectra()
    expect_identical(tic(sps), numeric())

    sps <- Spectra(DataFrame(msLevel = c(1L, 1L)))
    expect_identical(tic(sps), c(NA_real_, NA_real_))

    df <- DataFrame(msLevel = c(1L, 1L), totIonCurrent = c(5, 3))
    sps <- Spectra(df)
    expect_identical(tic(sps), c(5, 3))
    expect_identical(tic(sps, initial = FALSE), c(0, 0))

    df$intensity <- list(c(3, 3, 1), c(5, 3, 1))
    df$mz <- list(1:3, 1:3)
    sps <- Spectra(df)
    expect_identical(tic(sps, initial = FALSE), c(7, 9))
})

test_that("$, $<-, [[, [[<-,Spectra works", {
    sps <- Spectra()
    expect_identical(sps$msLevel, integer())
    expect_identical(sps[["msLevel"]], integer())
    expect_error(sps[[3]], "to be a character")
    expect_error(sps[["msLevel", j = 3]], "not supported")

    expect_error(sps$not_there, "No spectra variable")
    expect_error(sps[["not_there"]], "No spectra variable")

    sps <- Spectra(DataFrame(msLevel = c(1L, 2L), other_column = "a"))
    expect_identical(sps$msLevel, c(1L, 2L))
    expect_identical(sps$other_column, c("a", "a"))
    expect_identical(sps[["msLevel"]], c(1L, 2L))
    expect_identical(sps[["other_column"]], c("a", "a"))

    sps$second_col <- c(1, 4)
    expect_identical(sps$second_col, c(1, 4))
    expect_true(any(spectraVariables(sps) == "second_col"))

    df <- DataFrame(msLevel = c(1L, 2L), other_column = "a")
    df$mz <- list(1:3, 1:4)
    df$intensity <- list(c(3, 6, 3), c(56, 6, 3, 2))
    sps <- Spectra(df)

    sps$intensity <- list(c(4, 4, 4), c(2, 4, 6, 3))
    expect_equal(intensity(sps), NumericList(c(4, 4, 4), c(2, 4, 6, 3),
                                             compress = FALSE))

    sps[["msLevel"]] <- c(3L, 2L)
    expect_equal(msLevel(sps), c(3L, 2L))
    sps[["intensity"]] <- list(c(1, 2, 3), c(1, 2, 3, 4))
    expect_equal(intensity(sps), NumericList(c(1, 2, 3), c(1, 2, 3, 4),
                                             compress = FALSE))

    sps <- Spectra(sciex_mzr)
    sps$add_col <- "something"
    expect_true(all(sps$add_col == "something"))

    expect_error(sps$mz <- mz(sps))

    expect_error(sps$add_col <- c(1, 2), "has to be either 1 or")
})

#### ---------------------------------------------------------------------------
##
##                      FILTERING AND SUBSETTING
##
#### ---------------------------------------------------------------------------

test_that("[,Spectra works", {
    df <- DataFrame(msLevel = c(1L, 2L, 1L, 2L),
                    rtime = c(1, 2, 1, 2))
    sps <- Spectra(df)
    res <- sps[c(2, 4), ]
    expect_identical(msLevel(res), c(2L, 2L))
    expect_identical(sps, sps[])

    expect_error(sps[, 4], "columns is not")

    res <- sps[integer()]
    expect_true(is(res, "Spectra"))
    expect_true(length(res) == 0)

    sps <- Spectra(sciex_mzr)
    tmp <- sps[dataStorage(sps) == sciex_file[2], ]
    expect_true(all(dataStorage(tmp) == sciex_file[2]))
    expect_equal(unique(tmp$dataStorage), sciex_file[2])
    expect_equal(rtime(tmp), rtime(sps)[dataStorage(sps) == sciex_file[2]])
})

test_that("filterAcquisitionNum,Spectra works", {
    sps <- Spectra()
    res <- filterAcquisitionNum(sps, n = 3L)
    expect_equal(length(res), 0)
    expect_equal(length(res@processing), 1)

    sps <- Spectra(sciex_mzr)
    res <- filterAcquisitionNum(sps, n = 1:10, dataStorage = sciex_file[2])
    expect_equal(acquisitionNum(res),
                 c(1:sum(dataStorage(sps) == sciex_file[1]), 1:10))

    expect_error(filterAcquisitionNum(sps, dataStorage = 2), "type character")
    expect_error(filterAcquisitionNum(sps, dataOrigin = 2), "type character")
})

test_that("filterDataOrigin,Spectra works", {
    sps <- Spectra()
    res <- filterDataOrigin(sps)
    expect_true(length(res) == 0)
    expect_true(length(res@processing) == 1)

    expect_error(filterDataOrigin(sps, 3), "type character")

    sps <- Spectra(sciex_mzr)
    res <- filterDataOrigin(sps, dataOrigin = "2")
    expect_true(length(res) == 0)
    res <- filterDataOrigin(sps, sciex_file[1])
    expect_identical(rtime(res), rtime(sps)[1:931])
    expect_true(length(res@processing) > length(sps@processing))

    dorig <- rep(letters[1:7], each = length(sps)/7)
    dataOrigin(sps) <- dorig
    res <- filterDataOrigin(sps, dataOrigin = c("d", "a"))
    expect_equal(unique(dataOrigin(res)), c("d", "a"))
    expect_equal(rtime(res)[1:266], rtime(sps)[sps$dataOrigin == "d"])
    expect_equal(peaksData(res)[1:266],
                 SimpleList(sciex_pks[sps$dataOrigin == "d"]))
})

test_that("filterDataStorage,Spectra works", {
    sps <- Spectra()
    res <- filterDataStorage(sps)
    expect_true(length(res) == 0)
    expect_true(length(res@processing) == 1)

    expect_error(filterDataStorage(sps, 3), "type character")

    sps <- Spectra(sciex_mzr)
    res <- filterDataStorage(sps, "2")
    expect_true(length(res) == 0)
    res <- filterDataStorage(sps, sciex_file[2])
    expect_identical(rtime(res), rtime(sps)[dataStorage(sps) == sciex_file[2]])
    expect_true(length(res@processing) > length(sps@processing))
    expect_identical(peaksData(res),
                     SimpleList(sciex_pks[dataStorage(sps) == sciex_file[2]]))
})

test_that("filterEmptySpectra,Spectra works", {
    sps <- Spectra()
    res <- filterEmptySpectra(sps)
    expect_true(length(res) == 0)
    expect_true(length(res@processing) == 1)

    df <- DataFrame(msLevel = c(1L, 2L, 1L, 2L),
                    rtime = c(1, 2, 1, 2), centroided = TRUE)
    df$mz <- SimpleList(1:3, integer(), 1:5, integer())
    df$intensity <- SimpleList(c(4, 6, 3), numeric(), c(34, 2, 5, 9, 9), numeric())
    sps <- Spectra(df)

    res <- filterEmptySpectra(sps)
    expect_equal(length(res), 2)
    expect_equal(length(res@processing), 1)
    expect_equal(rtime(res), c(1, 1))

    sps <- filterIntensity(replaceIntensitiesBelow(
        sps, threshold = 20, value = NA_real_))
    res <- filterEmptySpectra(sps)
    expect_equal(length(res), 1)
    expect_equal(rtime(res), 1)
    expect_equal(length(res@processing), 3)

    sps <- filterIntensity(replaceIntensitiesBelow(
        sps, threshold = 50, value = NA_real_))
    res <- filterEmptySpectra(sps)
    expect_equal(length(res), 0)

    sps <- Spectra(sciex_mzr)
    res <- filterEmptySpectra(sps)
    expect_equal(rtime(res), rtime(sps))
})

test_that("filterIsolationWindow,Spectra works", {
    sps <- Spectra()
    res <- filterIsolationWindow(sps)
    expect_true(is(res, "Spectra"))
    expect_true(length(res@processing) == 1)

    sps <- Spectra(sciex_mzr)
    res <- filterIsolationWindow(sps, 123.323)
    expect_true(length(res) == 0)

    sps <- Spectra(tmt_mzr)
    res <- filterIsolationWindow(sps, 544)
    expect_true(length(res) == 3)
    expect_true(all(isolationWindowLowerMz(res) < 544))
    expect_true(all(isolationWindowUpperMz(res) > 544))
    expect_true(all(precursorMz(res) < 545 & precursorMz(res) > 543))
})

test_that("filterMsLevel,Spectra works", {
    sps <- Spectra()
    res <- filterMsLevel(sps)
    expect_true(length(res) == 0)
    expect_true(length(res@processing) == 1)

    sps <- Spectra(sciex_mzr)
    res <- filterMsLevel(sps, 2L)
    expect_true(length(res) == 0)

    sps <- Spectra(tmt_mzr)
    expect_true(all(1:2 %in% msLevel(sps)))
    res <- filterMsLevel(sps, 1L)
    expect_false(all(1:2 %in% msLevel(res)))
    expect_true(all(msLevel(res) == 1))

    res <- filterMsLevel(sps, c(2L, 4L))
    expect_false(all(1:2 %in% msLevel(res)))
    expect_true(all(msLevel(res) == 2))
})

test_that("filterPolarity,Spectra works", {
    sps <- Spectra()
    res <- filterPolarity(sps)
    expect_true(length(res) == 0)
    expect_true(length(res@processing) == 1)

    sps <- Spectra(sciex_mzr)
    res <- filterPolarity(sps, polarity = c(2, 0))
    expect_true(length(res) == 0)

    res <- filterPolarity(sps, 1)
    expect_true(all(polarity(res) == 1))
    expect_equal(rtime(res), rtime(sps))
    expect_true(length(res@processing) == 1)
})

test_that("filterPrecursorMzRange,Spectra works", {
    sps <- Spectra()
    res <- filterPrecursorMzRange(sps)
    expect_true(is(res, "Spectra"))
    expect_true(length(res@processing) == 1)

    sps <- Spectra(tmt_mzr)
    res <- filterPrecursorMzRange(sps, mz = 544.75)
    expect_true(length(res) == 0)
    res <- filterPrecursorMzRange(sps, mz = 544.75 + c(-1, 1) * ppm(544.75, 40))
    expect_true(length(res) == 2)
})

test_that("filterPrecursorCharge,Spectra works", {
    sps <- Spectra()
    res <- filterPrecursorCharge(sps)
    expect_true(is(res, "Spectra"))
    expect_true(length(res@processing) == 1)

    sps <- Spectra(tmt_mzr)
    res <- filterPrecursorCharge(sps, z = 0L)
    expect_true(length(res) == 0)

    res2 <- filterPrecursorCharge(sps, z = 2)
    res3 <- filterPrecursorCharge(sps, z = 3)
    res23 <- filterPrecursorCharge(sps, z = 2:3)
    expect_true(length(res2) == 300)
    expect_true(length(res3) == 128)
    expect_true(length(res23) == (length(res2) + length(res3)))
})


test_that("filterPrecursorScan,Spectra works", {
    sps <- Spectra()
    res <- filterPrecursorScan(sps, 3)
    expect_true(length(res) == 0)
    expect_true(is(res, "Spectra"))
    expect_true(length(res@processing) == 1)

    sps <- Spectra(tmt_mzr)
    res <- filterPrecursorScan(sps, c(1087L, 1214L))
    expect_true(sum(msLevel(res) == 1) == 2)
    expect_true(all(c(1087L, 1214L) %in% acquisitionNum(res)))

    sps <- c(sps, Spectra(sciex_mzr))
    res <- filterPrecursorScan(sps, 1057)
    expect_equal(acquisitionNum(res), c(1054L, 1057L))
})

test_that("filterRt,Spectra works", {
    sps <- Spectra()
    res <- filterRt(sps, c(1, 2))
    expect_true(is(res, "Spectra"))
    expect_true(length(res) == 0)
    expect_true(length(res@processing) == 1)

    sps <- Spectra(sciex_mzr)
    res <- filterRt(sps, rt = c(100, 120))
    expect_true(all(rtime(res) >= 100 & rtime(res) <= 120))
    expect_error(filterRt(sps, rt = c(100)))
    expect_error(filterRt(sps, rt = c(120, 100)))
    expect_error(filterRt(sps, rt = c("100", "120")))
    expect_error(filterRt(sps, rt = c(100, 120), msLevel. = "1"))

    res <- filterRt(sps, rt = c(100, 120), msLevel = 2L)
    expect_equal(rtime(res), rtime(sps))
})

test_that("filterRanges, Spectra works", {
    filt_spectra <- filterRanges(sps_dia, spectraVariables = c("rtime",
                                                               "precursorMz",
                                                               "peaksCount"),
                                 ranges = c(30, 350, 200,500, 350, 600))
    # test does not accept logical
    logical_test <- spectraVariables(sps_dia) %in% c("rtime", "precursorMz",
                                                     "peaksCount")
    expect_error(filterRanges(sps_dia, spectraVariables = logical_test,
                              ranges = c(30, 350, 200,500, 350, 600)),
                 "character")
    # do not accept values other than numerical
    fls <- unique(dataOrigin(sps_dia))
    expect_error(filterRanges(sps_dia, spectraVariables = "dataOrigin",
                 ranges = c(fls[1], fls[1])), "numerical")
    # test too many variables
    expect_error(filterRanges(sps_dia, spectraVariables = c("rtime",
                                                            "precursorMz",
                                                            "peaksCount",
                                                            "lowMZ"),
                              ranges = c(30, 350, 200,500, 350, 600)), "Length")
    # test too many ranges
    expect_error(filterRanges(sps_dia, spectraVariables = c("rtime",
                                                            "precursorMz",
                                                            "peaksCount"),
                              ranges = c(30, 350, 200,500, 350, 600, 20, 800)),
                 "Length")
    expect_true(length(sps_dia) > length(filt_spectra))
    # test does not accept variables not in spectraData
    expect_error(filterRanges(sps_dia, spectraVariables = c("rtime",
                                                            "precursorMz",
                                                            "peaksCount",
                                                            "fakeVar"),
                              ranges = c(30, 350, 200,500, 350, 600, 20, 800)),
                 "not available")
    # test same results as specific filtering functions
    spe_fct <- filterPrecursorMzRange(sps_dia, c(200,500))
    range_fct <- filterRanges(sps_dia, spectraVariables = "precursorMz",
                              ranges = c(200,500))
    expect_equal(length(spe_fct), length(range_fct))
    # test any match
    ranges <- c(30, 60, 200, 250)
    filt_spectra <- filterRanges(sps_dia, spectraVariables = c("rtime", "rtime"),
                    ranges = ranges, match = "any")
    expect_true(all(range(rtime(filt_spectra)) <= 250 &
                        range(rtime(filt_spectra)) >= 30))
})

test_that("filterValues, Spectra works", {
    # Not testing for the same sanity checks as filterRanges
    ## expect error
    fls <- unique(dataOrigin(sps_dia))
    expect_error(filterValues(sps_dia, spectraVariables = "dataOrigin",
                 values = fls[1]), "numerical")
    ## test recycling
    filt_spectra <- filterValues(sps_dia, spectraVariables = c("rtime",
                                                               "precursorMz",
                                                               "peaksCount"),
                                 values = c(200, 400, 350),
                                 tolerance = c(100, 100, 100),
                                 ppm = c(0 ,30, 0))
    filt_recycle <- filterValues(sps_dia, spectraVariables = c("rtime",
                                                               "precursorMz",
                                                               "peaksCount"),
                                 values = c(200, 400, 350),
                                 tolerance = 100,
                                 ppm = c(0, 40, 0))
    expect_equal(length(filt_spectra), length(filt_recycle))
    expect_true(length(sps_dia) > length(filt_spectra))
    #' expect warning
    expect_warning(filterValues(sps_dia, spectraVariables = c("rtime",
                                                              "precursorMz",
                                                              "peaksCount"),
                                values = c(200, 400, 350),
                                tolerance = 100), "recycled")
    #' test same results as filterPrecursorMzValues
    spe_fct <- filterPrecursorMzValues(sps_dia, mz = 300, ppm = 20, tolerance = 10)
    values_fct <- filterValues(sps_dia, spectraVariables = "precursorMz",
                               values = 300, ppm = 20, tolerance = 10)
    expect_equal(length(spe_fct), length(values_fct))

    # test any
    values <- c(200, 400)
    filt_spectra <- filterValues(sps_dia, spectraVariables = c("rtime", "rtime"),
                                 values = values, match = "any",
                                 tolerance = 100)
    expect_true(all(range(rtime(filt_spectra)) <= 500 &
                        range(rtime(filt_spectra)) >= 100))
})

#### ---------------------------------------------------------------------------
##
##                      DATA MANIPULATION METHODS
##
#### ---------------------------------------------------------------------------

test_that("bin,Spectra works", {
    sps <- Spectra(tmt_mzr)
    pks <- peaksData(sps)
    res <- bin(sps, binSize = 2, zero.rm = FALSE)
    expect_true(length(res@processingQueue) == 1)
    res1 <- bin(sps, msLevel = 1, binSize = 2, zero.rm = FALSE)

    expect_identical(peaksData(res1)[res1$msLevel == 2],
                     pks[sps$msLevel == 2])

    mzr <- range(unlist(mz(sps)))
    brks <- MsCoreUtils:::.fix_breaks(
                              seq(floor(mzr[1]), ceiling(mzr[2]), by = 2), mzr)
    res1 <- bin(sps, msLevel = 1, breaks = brks, zero.rm = FALSE)
    res1_pks <- peaksData(res1)
    res_pks <- peaksData(res)
    expect_identical(res1_pks[res1$msLevel == 1],
                     res_pks[res$msLevel == 1])
    expect_true(all(lengths(res_pks) != lengths(pks)))

    expect_warning(res <- bin(sps, msLevel = 3))
    expect_identical(res, sps)
})

test_that("filterIntensity,Spectra works", {
    sps <- Spectra()
    res <- filterIntensity(sps)
    expect_true(length(res@processingQueue) == 1)
    expect_equal(res@processingQueue[[1]],
                 ProcessingStep(.peaks_filter_intensity,
                                list(intensity = c(0, Inf),
                                     msLevel = integer())))

    res <- filterIntensity(sps, msLevel = 2L)
    expect_true(length(res@processingQueue) == 1)
    expect_equal(res@processingQueue[[1]],
                 ProcessingStep(.peaks_filter_intensity,
                                list(intensity = c(0, Inf),
                                     msLevel = 2L)))

    res <- filterIntensity(Spectra(sciex_mzr), intensity = c(500, 9000))
    ints <- unlist(intensity(res), use.names = FALSE)
    expect_true(all(ints >= 500 & ints <= 9000))

    expect_error(filterIntensity(Spectra(sciex_mzr), c(1, 2, 3)), "limit")

    ## With `intensity` being a function.
    sps <- Spectra()
    res <- filterIntensity(sps, intensity = function(x) x > mean(x))
    expect_true(length(res@processingQueue) == 1)
    expect_true(length(intensity(res)) == 0)

    expect_error(filterIntensity(sps, intensity = TRUE), "numeric or a fun")

    df <- DataFrame(msLevel = c(1L, 2L), fromFile = 1L)
    df$mz <- list(1:4, 1:5)
    df$intensity <- list(1:4, 1:5)
    sps <- Spectra(df)

    res <- filterIntensity(sps, intensity = function(x) x > max(x)/2)
    expect_equal(intensity(res)[[1L]], c(3, 4))
    expect_equal(intensity(res)[[2L]], c(3, 4, 5))
    res <- filterIntensity(sps, intensity = function(x) x > max(x)/2,
                           msLevel = 1L)
    expect_equal(intensity(res)[[1L]], c(3, 4))
    expect_equal(intensity(res)[[2L]], c(1, 2, 3, 4, 5))

    ## Passing additional parameters.
    filt_fun <- function(x, thresh = 1) x > thresh
    res <- filterIntensity(sps, intensity = filt_fun)
    expect_equal(intensity(res)[[1L]], c(2:4))
    expect_equal(intensity(res)[[2L]], c(2:5))

    res <- filterIntensity(sps, intensity = filt_fun, thresh = 2)
    expect_equal(intensity(res)[[1L]], c(3:4))
    expect_equal(intensity(res)[[2L]], c(3:5))
})

test_that("compareSpectra works", {
    sps <- Spectra(sciex_hd5[1:20])
    sps <- setBackend(sps, MsBackendDataFrame())

    res <- compareSpectra(sps[c(1, 20)], sps[15:20])
    expect_true(nrow(res) == 2)
    expect_true(ncol(res) == 6)
    expect_equal(res[2, 6], 1)
    expect_true(all(res > 0.8))

    spectraNames(sps) <- seq_along(sps)
    res <- compareSpectra(sps[c(1, 20)], sps[15:20])
    expect_equal(rownames(res), c("1", "20"))
    expect_equal(colnames(res), as.character(15:20))

    res <- compareSpectra(sps[1], sps[15:20])
    expect_false(is.matrix(res))
    res <- compareSpectra(sps[1], sps[15:20], SIMPLIFY = FALSE)
    expect_true(is.matrix(res))

    res <- compareSpectra(sps[15:20], sps[4])
    expect_false(is.matrix(res))

    res <- compareSpectra(Spectra(), sps)
    expect_true(is.matrix(res))
    expect_equal(nrow(res), 0)

    res <- compareSpectra(sps, Spectra())
    expect_true(is.matrix(res))
    expect_equal(ncol(res), 0)

    ## y missing
    res <- compareSpectra(sps[2:5])
    expect_equal(nrow(res), ncol(res))
    expect_equal(unname(diag(res)), rep(1, 4))

    res <- compareSpectra(sps[1])
    expect_equal(res, 1)

    ## FUN
    cor_fun <- function(x, y, ...) {
        cor(x[, 2], y[, 2], use = "pairwise.complete.obs")
    }
    res <- compareSpectra(sps[1], sps[2])
    res_2 <- compareSpectra(sps[1], sps[2], FUN = cor_fun)
    expect_true(res < res_2)

    ## Empty spectra:
    sps2 <- Spectra()
    res <- compareSpectra(sps2)
    expect_true(is.matrix(res))
    expect_true(nrow(res) == 0)
    expect_true(ncol(res) == 0)

    res <- compareSpectra(sps, sps2)
    expect_true(is.matrix(res))
    expect_true(ncol(res) == 0)
    expect_true(nrow(res) == length(sps))

    ## Check compareSpectra with GNPS score.
    df <- DataFrame(msLevel = 2L, precursorMz = c(488.358, 356.28))
    df$intensity <- IRanges::NumericList(
                                 c(7114.173828, 1001.093994, 770.18103,
                                   13194.55957, 6228.618164, 8398.287109,
                                   1492.104004, 5111.643066, 2547.698975,
                                   1840.756958, 245.207001, 3336.004883,
                                   313.119995, 626.439026, 4527.278809,
                                   461.109985, 682.629028, 8239.036133,
                                   43.747002, 16.875999, 240.197998,
                                   276.912994, 65.282997, 31.399, 14.15),
                                 c(7689.937012, 1007.257996, 4579.356934,
                                   8001.039062, 2327.36792, 881.854004,
                                   446.681, 6326.22998, 497.213013,
                                   17089.107422, 153.132996, 17.469999,
                                   53.928001, 93.917, 5.337, 11.661,
                                   3.787, 73.805), compress = FALSE)
    df$mz <- IRanges::NumericList(
                          c(57.068001, 74.096001, 87.043999, 89.059998,
                            121.065002, 133.085999, 147.080002, 165.091003,
                            177.112, 209.117004, 221.139008, 233.190002,
                            253.143005, 271.153992, 277.216003, 315.181,
                            321.243011, 359.207001, 365.21701, 383.179993,
                            401.166992, 419.177002, 427.303986, 442.194,
                            471.330994),
                          c(57.07, 74.096001, 89.059998, 121.065002,
                            133.085999, 139.074997, 147.080002, 165.091003,
                            209.117996, 227.128006, 233.190002, 250.901001,
                            268.912994, 277.216003, 303.231995, 309.901001,
                            315.194, 321.239014), compress = FALSE)
    sp <- Spectra(df)
    res <- compareSpectra(sp, MAPFUN = joinPeaksGnps, FUN = MsCoreUtils::gnps,
                          tolerance = 0.1)
    expect_equal(round(res[1, 2], 2), 0.86)
    res <- compareSpectra(sp[1], sp[2], MAPFUN = joinPeaksGnps,
                          FUN = MsCoreUtils::gnps, tolerance = 0.1)
    expect_equal(round(res, 2), 0.86)
})

test_that("pickPeaks,Spectra works", {
    sps <- Spectra()
    expect_error(pickPeaks(sps, halfWindowSize = 1), "integer")
    expect_error(pickPeaks(sps, halfWindowSize = 1L:2L), "length 1")
    expect_error(pickPeaks(sps, halfWindowSize = -1L), "> 0")
    expect_error(pickPeaks(sps, method = "foo"), "MAD")
    expect_error(pickPeaks(sps, snr = "foo"), "numeric")
    expect_error(pickPeaks(sps, snr = 1L:2L), "length 1")
    expect_error(pickPeaks(sps, snr = -1L), ">= 0")
    expect_error(pickPeaks(sps, k = 1), "integer")
    expect_error(pickPeaks(sps, k = 1L:2L), "length 1")
    expect_error(pickPeaks(sps, k = -1L), ">= 0")
    expect_error(pickPeaks(sps, descending = NA), "TRUE or FALSE")
    expect_error(pickPeaks(sps, descending = c(TRUE, TRUE)), "TRUE or FALSE")
    expect_error(pickPeaks(sps, threshold = "foo"), "numeric")
    expect_error(pickPeaks(sps, threshold = 1L:2L), "length 1")
    expect_error(pickPeaks(sps, threshold = -1L), ">= 0")
    expect_error(pickPeaks(sps, threshold = 2L), "<= 1")

    res <- pickPeaks(sps)
    expect_true(length(res@processingQueue) == 1)
    expect_equal(res@processingQueue[[1]],
                 ProcessingStep(.peaks_pick,
                                list(halfWindowSize = 2L, method = "MAD",
                                     snr = 0, k = 0L, descending = FALSE,
                                     threshold = 0L, msLevel = integer())))
    expect_match(res@processing,
                 "Peak picking with MAD noise estimation, hws = 2, snr = 0 \\[")
    res <- pickPeaks(sps, k = 2L)
    expect_match(res@processing,
                 paste0("Peak picking with MAD noise estimation, hws = 2, ",
                        "snr = 0 and centroid refinement \\["))

    sps <- Spectra(sciex_mzr)
    expect_warning(expect_equal(pickPeaks(sps, msLevel. = 3), sps))

    res <- pickPeaks(sps)
    pks_res <- lapply(sciex_pks, .peaks_pick, spectrumMsLevel = 1L,
                      centroided = FALSE)
    expect_identical(peaksData(res), SimpleList(pks_res))
    expect_true(all(centroided(res)))

    ## Check that ... works
    res2 <- pickPeaks(sps, method = "SuperSmoother", snr = 2, k = 0L,
                      descending = FALSE, threshold = 0, span = 0.2)
    expect_true(length(intensity(res2[1L])[[1L]]) <
                length(intensity(res[1L])[[1L]]))
    res3 <- pickPeaks(sps, method = "SuperSmoother", snr = 2, k = 0L,
                      descending = FALSE, threshold = 0)
    expect_true(length(intensity(res2[1L])[[1L]]) <
                length(intensity(res3[1L])[[1L]]))

    ## Check that m/z refinement works
    a <- data.frame(msLevel = 1L, rtime = 1.1)
    a$mz <- list(
        c(34.1, 56.3, 100.1, 100.101, 100.105, 100.2, 200.2,
          200.205, 300.2, 350.2))
    a$intensity <- list(
        c(3.1, 2.3, 100, 200, 130, 120, 400.1,
          100, 40, 12))
    a <- Spectra(a)
    ## Default: report the max peak.
    res <- pickPeaks(a, halfWindowSize = 2L)
    expect_equal(unlist(intensity(res)), c(200, 400.1))
    expect_equal(unlist(mz(res)), c(100.101, 200.2))
    ## Refine the m/z
    res <- pickPeaks(a, halfWindowSize = 2L, k = 2L)
    expect_equal(res$mz[[1L]][1L],
                 weighted.mean(c(56.3, 100.1, 100.101, 100.105, 100.2),
                               c(2.3, 100, 200, 130, 120)))
})

test_that("smooth,Spectra works", {
    sps <- Spectra()
    expect_error(smooth(sps, halfWindowSize = 1), "integer")
    expect_error(smooth(sps, halfWindowSize = 1L:2L), "length 1")
    expect_error(smooth(sps, halfWindowSize = -1L), "> 0")
    expect_error(smooth(sps, method = "foo"), "MovingAverage")

    res <- smooth(sps)
    expect_true(length(res@processingQueue) == 1)
    expect_equal(res@processingQueue[[1]],
                 ProcessingStep(.peaks_smooth,
                                list(halfWindowSize = 2L,
                                     coef = matrix(0.2, nrow = 5, ncol = 5),
                                     msLevel = integer())))
    expect_match(res@processing,
                 "Spectra smoothing with MovingAverage, hws = 2 \\[")

    sps <- Spectra(sciex_mzr)
    expect_equal(smooth(sps, msLevel. = 3), sps)

    res <- smooth(sps)
    pks_res <- lapply(sciex_pks, .peaks_smooth,
                      spectrumMsLevel = 1L, coef = coefMA(2L))
    expect_identical(peaksData(res), SimpleList(pks_res))
})

test_that("replaceIntensitiesBelow,Spectra works", {
    sps <- Spectra()
    res <- replaceIntensitiesBelow(sps, threshold = 10)
    expect_true(length(res@processingQueue) == 1)
    expect_equal(res@processingQueue[[1]],
                 ProcessingStep(.peaks_replace_intensity,
                                list(threshold = 10, value = 0,
                                     msLevel = integer())))

    expect_error(replaceIntensitiesBelow(sps, threshold = "b"), "numeric")

    sps <- Spectra(sciex_mzr)
    centroided(sps) <- TRUE
    res <- replaceIntensitiesBelow(sps, threshold = 5000)
    pks_res <- lapply(sciex_pks, .peaks_replace_intensity, threshold = 5000,
                      spectrumMsLevel = 1L, centroided = TRUE)
    expect_identical(peaksData(res), SimpleList(pks_res))
})

test_that("spectrapply,Spectra works", {
    sps <- Spectra(sciex_mzr)[c(1:3, 1400:1410)]
    rts <- spectrapply(sps, rtime)
    expect_equal(unlist(rts, use.names = FALSE), rtime(sps))

    expect_equal(unname(split(sps, 1:length(sps))), unname(spectrapply(sps)))

    ## test on a mzR backend using intensities.
    myFun <- function(x, add) {
        mean(intensity(x)[[1]]) + add
    }
    res <- spectrapply(sps, FUN = myFun, add = 3)
    ints <- intensity(sps)
    expect_equal(unlist(res, use.names = FALSE),
                 vapply(ints, mean, numeric(1)) + 3)

    ## Same after replaceIntensitiesBelow and clean.
    sps <- filterIntensity(replaceIntensitiesBelow(sps, t = 4000),
                           intensity = 0.1)
    res <- spectrapply(sps, FUN = function(x) mean(x$intensity[[1]]))
    expect_equal(unlist(res, use.names = FALSE),
                 vapply(intensity(sps), mean, numeric(1)))

    ## chunkify
    res <- spectrapply(Spectra(sciex_mzr), lengths, chunkSize = 100)
    expect_equal(res, lengths(sciex_pks) / 2)
})

test_that("split,Spectra works", {
    sps <- Spectra(sciex_mzr)
    res <- split(sps, f = sps$dataStorage)
    expect_identical(res, split.default(sps, f = sps$dataStorage))
})

test_that("containsMz,Spectra works", {
    spd <- DataFrame(msLevel = c(2L, 2L, 2L), rtime = c(1, 2, 3))
    spd$mz <- list(c(12, 14, 45, 56), c(14.1, 34, 56.1), c(12.1, 14.15, 34.1))
    spd$intensity <- list(c(10, 20, 30, 40), c(11, 21, 31), c(12, 22, 32))
    sps <- Spectra(spd)

    res <- containsMz(sps)
    expect_true(all(is.na(res)))

    res <- containsMz(sps, NA)
    expect_true(all(is.na(res)))

    res <- containsMz(sps, c(14.15), which = "any")
    expect_equal(res, c(FALSE, FALSE, TRUE))

    res_2 <- containsMz(sps, c(14.15), which = "any",
                        BPPARAM = MulticoreParam(2))
    expect_equal(res, res_2)
    ## Check that unsplit works.
    sps@backend$dataStorage <- c("3", "1", "2")
    res_2 <- containsMz(sps, c(14.15))
    expect_equal(res, res_2)
})

test_that("containsNeutralLoss,Spectra works", {
    spd <- DataFrame(msLevel = c(2L, 2L, 2L), rtime = c(1, 2, 3),
                     precursorMz = c(NA, 38, 16))
    spd$mz <- list(c(12, 14, 45, 56), c(14.1, 34, 56.1), c(12.1, 14.15, 34.1))
    spd$intensity <- list(c(10, 20, 30, 40), c(11, 21, 31), c(12, 22, 32))
    sps <- Spectra(spd)

    res <- containsNeutralLoss(sps, neutralLoss = 4, BPPARAM = SerialParam())
    expect_equal(res, c(NA, TRUE, FALSE))
    res <- containsNeutralLoss(sps, neutralLoss = 4, BPPARAM = SerialParam(),
                               tolerance = 0.1)
    expect_equal(res, c(NA, TRUE, TRUE))

    ## Compare with splitting/parallel.
    res <- containsNeutralLoss(sps, neutralLoss = 4, BPPARAM = SerialParam())
    expect_equal(res, c(NA, TRUE, FALSE))

    res_2 <- containsNeutralLoss(sps, neutralLoss = 4, BPPARAM = MulticoreParam())
    expect_equal(res, res_2)

    sps@backend$dataStorage <- c("3", "1", "2")
    res_2 <- containsNeutralLoss(sps, neutralLoss = 4, BPPARAM = MulticoreParam())
    expect_equal(res, res_2)
})

test_that("reset,Spectra works", {
    spd <- DataFrame(msLevel = c(2L, 2L, 2L), rtime = c(1, 2, 3),
                     precursorMz = c(NA, 38, 16))
    spd$mz <- list(c(12, 14, 45, 56), c(14.1, 34, 56.1), c(12.1, 14.15, 34.1))
    spd$intensity <- list(c(10, 20, 30, 40), c(11, 21, 31), c(12, 22, 32))
    sps <- Spectra(spd)

    res <- reset(sps)
    expect_equal(mz(res), mz(sps))

    sps_mod <- filterIntensity(sps, intensity = 29)
    res <- reset(sps_mod)
    expect_equal(mz(res), mz(sps))
})

test_that("export,Spectra works", {
    spd <- DataFrame(msLevel = c(2L, 2L, 2L), rtime = c(1, 2, 3),
                     precursorMz = c(NA, 38, 16))
    spd$mz <- list(c(12, 14, 45, 56), c(14.1, 34, 56.1), c(12.1, 14.15, 34.1))
    spd$intensity <- list(c(10, 20, 30, 40), c(11, 21, 31), c(12, 22, 32))
    sps <- Spectra(spd)

    fl <- tempfile()
    expect_error(export(sps, backend = MsBackendDataFrame(),
                        file = fl), "MsBackendDataFrame does not")
    expect_warning(
        export(sps, backend = MsBackendMzR(), file = fl, copy = TRUE),
        "Original data file not found")
})

test_that("filterMzRange,Spectra works", {
    spd <- DataFrame(msLevel = c(2L, 2L, 2L), rtime = c(1, 2, 3),
                     precursorMz = c(NA, 38, 16))
    spd$mz <- list(c(12, 14, 45, 56), c(14.1, 34, 56.1), c(12.1, 14.15, 34.1))
    spd$intensity <- list(c(10, 20, 30, 40), c(11, 21, 31), c(12, 22, 32))
    sps <- Spectra(spd)

    expect_warning(res <- filterMzRange(sps, msLevel = 1L), "not available")
    expect_equal(mz(res), mz(sps))

    res <- filterMzRange(sps)
    expect_equal(mz(res), mz(sps))

    res <- filterMzRange(sps, mz = c(200, 400))
    expect_true(all(lengths(mz(res)) == 0))

    res <- filterMzRange(sps, mz = c(40, 60))
    expect_equal(mz(res)[[1L]], c(45, 56))
    expect_equal(unname(mz(res)[[2L]]), 56.1)
    expect_true(length(mz(res)[[3L]]) == 0)

    ## Remove
    res <- filterMzRange(sps, mz = c(200, 400), keep = FALSE)
    expect_equal(mz(res), mz(sps))

    res <- filterMzRange(sps, mz = c(12, 15), keep = FALSE)
    expect_equal(mz(res)[[1L]], c(45, 56))
    expect_equal(mz(res)[[2L]], c(34, 56.1))
    expect_equal(unname(mz(res)[[3L]]), 34.1)
})

test_that("filterMzValue,Spectra works", {
    spd <- DataFrame(msLevel = c(2L, 2L, 2L), rtime = c(1, 2, 3),
                     precursorMz = c(NA, 38, 16))
    spd$mz <- list(c(12, 14, 45, 56), c(14.1, 34, 56.1), c(12.1, 14.15, 34.1))
    spd$intensity <- list(c(10, 20, 30, 40), c(11, 21, 31), c(12, 22, 32))
    sps <- Spectra(spd)

    res <- filterMzValues(sps, mz = 56)
    expect_equal(unname(mz(res)[[1L]]), 56)
    expect_true(length(mz(res)[[2L]]) == 0)
    expect_true(length(mz(res)[[3L]]) == 0)

    res <- filterMzValues(sps, mz = c(56, 12), tolerance = c(0.2))
    expect_equal(mz(res)[[1L]], c(12, 56))
    expect_equal(unname(mz(res)[[2L]]), 56.1)
    expect_equal(unname(mz(res)[[3L]]), 12.1)

    expect_error(filterMzValues(sps, mz = c(56, 12), tolerance = c(1, 2, 3)),
                 "length 1")
    expect_error(filterMzValues(sps, mz = c(56, 12), ppm = c(1, 2, 3)),
                 "length 1")

    ## remove
    res <- filterMzValues(sps, mz = 56, keep = FALSE)
    expect_equal(mz(res)[[1L]], mz(sps)[[1L]][-4])
    res <- filterMzValues(sps, mz = 56, keep = FALSE, tolerance = 0.1)
    expect_equal(mz(res)[[1L]], mz(sps)[[1L]][-4])
    expect_equal(mz(res)[[2L]], mz(sps)[[2L]][-3])

    res <- filterMzValues(sps, mz = c(1243, 244), keep = FALSE)
    expect_equal(mz(res), mz(sps))

    ## Second set of tests
    spd$mz <- list(c(12, 14, 45, 45.1, 45.2, 45.3, 56),
                   c(14.1, 34, 45.1, 45.2, 56.1),
                   c(12.1, 14.15, 34.1, 45.4))
    spd$intensity <- list(c(10, 20, 30, 40, 50, 40, 30),
                          c(11, 21, 31, 100, 100),
                          c(12, 22, 32, 100))
    sps <- Spectra(spd)
    res <- filterMzValues(sps, mz = 45, tolerance = 0.3, keep = FALSE)
    expect_equal(mz(res)[[1L]], c(12, 14, 56))
    expect_equal(mz(res)[[2L]], c(14.1, 34, 56.1))
    expect_equal(mz(res)[[3L]], c(12.1, 14.15, 34.1, 45.4))

    res <- filterMzValues(sps, mz = 45, tolerance = 0.3, keep = TRUE)
    expect_equal(mz(res)[[1L]], c(45, 45.1, 45.2, 45.3))
    expect_equal(mz(res)[[2L]], c(45.1, 45.2))
    expect_equal(mz(res)[[3L]], numeric())

    ## Multiple values
    res <- filterMzValues(sps, mz = c(56, 45), tolerance = 0.3, keep = FALSE)
    expect_equal(mz(res)[[1L]], c(12, 14))
    expect_equal(mz(res)[[2L]], c(14.1, 34))
    expect_equal(mz(res)[[3L]], c(12.1, 14.15, 34.1, 45.4))

    res <- filterMzValues(sps, mz = c(56, 45), tolerance = 0.3, keep = TRUE)
    expect_equal(mz(res)[[1L]], c(45, 45.1, 45.2, 45.3, 56))
    expect_equal(mz(res)[[2L]], c(45.1, 45.2, 56.1))
    expect_equal(mz(res)[[3L]], numeric())
})

test_that("dropNaSpectraVariables works with MsBackendMzR", {
    sps <- Spectra()
    res <- dropNaSpectraVariables(sps)
    expect_equal(spectraVariables(sps), spectraVariables(res))

    sps <- Spectra(sciex_mzr)
    res <- dropNaSpectraVariables(sps)
    expect_equal(mz(res[1]), mz(sps[1]))
    expect_true(length(spectraVariables(res)) <
                length(spectraVariables(sps)))

    df <- DataFrame(msLevel = c(1L, 1L), rtime = c(1.2, 1.3), centroided = TRUE)
    df$mz <- list(1:10, 1:10)
    df$intensity <- list(c(0, 0, 1, 6, 3, 0, 0, 9, 1, 0),
                         c(9, 6, 0, 0, 3, 0, 0, 0, 3, 2))
    sps <- Spectra(df)

    res <- dropNaSpectraVariables(sps)
    expect_equal(mz(res), mz(sps))
    expect_equal(intensity(res), intensity(sps))
    expect_equal(spectraVariables(res), spectraVariables(sps))

    sps$other_col <- NA
    res <- dropNaSpectraVariables(sps)
    expect_equal(mz(res), mz(sps))
    expect_true(length(spectraVariables(res)) < length(spectraVariables(sps)))
})

test_that("show,Spectra works", {
    be <- backendInitialize(MsBackendDataFrame(), DataFrame(msLevel = c(1L, 2L),
                                                            fromFile = 1L))
    sps <- Spectra(backend = be)
    expect_output(show(sps), "MsBackendDataFrame")

    sps@processing <- c("a", "b", "c", "d", "e")
    expect_output(show(sps), "2 more processings.")
})

test_that("filterFourierTransformArtefacts,Spectra", {
    data(fft_spectrum)
    a <- filterFourierTransformArtefacts(fft_spectrum)
    expect_true(is(a, "Spectra"))
    expect_true(lengths(a)[[1L]] < lengths(fft_spectrum)[[1L]])
    b <- filterFourierTransformArtefacts(fft_spectrum, halfWindowSize = 0.2)
    expect_true(lengths(b)[[1L]] < lengths(a)[[1L]])
})

test_that("peaksVariables,Spectra works", {
    sps <- Spectra(sciex_mzr)
    expect_equal(peaksVariables(sps), c("mz", "intensity"))
})

test_that("coreSpectraVariables works", {
    expect_equal(coreSpectraVariables(), .SPECTRA_DATA_COLUMNS)
})

test_that("uniqueMsLevels,Spectra works", {
    res <- uniqueMsLevels(sciex_mzr)
    expect_equal(res, unique(msLevel(sciex_mzr)))

    expect_equal(res, uniqueMsLevels(Spectra(sciex_mzr)))
})

with_parameters_test_that("peaks variables and filtering properly works", {
    test_df <- DataFrame(msLevel = c(1L, 2L, 2L), scanIndex = 4:6)
    test_df$mz <- list(c(1.1, 1.3, 1.5),
                       c(4.1, 5.1),
                       c(1.6, 1.7, 1.8, 1.9))
    test_df$intensity <- list(c(45.1, 34, 12),
                              c(234.4, 1333),
                              c(42.1, 34.2, 65, 6))
    test_df$pk_ann <- list(c(NA, NA, "C12H2"),
                           c("A", "B"),
                           c("D", "E", "F", "G"))
    s <- Spectra(test_df, source = bcknd,
                 peaksVariables = c("mz", "intensity", "pk_ann"))
    expect_equal(peaksVariables(s), c("mz", "intensity", "pk_ann"))
    expect_true(!any(spectraVariables(s) %in% peaksVariables(s)))

    spd <- spectraData(s)
    expect_true(!any(colnames(spd) %in% peaksVariables(s)))
    pkd <- peaksData(s)
    expect_equal(colnames(pkd[[1]]), c("mz", "intensity"))
    expect_equal(s$pk_ann, test_df$pk_ann)
    expect_equal(s$mz, spectraData(s, columns = "mz")$mz)
    expect_equal(test_df$pk_ann,
                 spectraData(s, columns = c("rtime", "pk_ann"))$pk_ann)

    ########
    ## Filter peaks. Have to ensure that ALL peak variables get subset properly
    sf <- filterIntensity(s, intensity = 34.1)
    expect_equal(mz(sf),
                 NumericList(list(c(mz=1.1), c(4.1, 5.1), c(1.6, 1.7, 1.8)),
                             compress = FALSE))
    expect_equal(mz(sf), sf$mz)

    ## peaksData
    expect_equal(lengths(sf), c(1, 2, 3))
    pkd <- peaksData(sf)
    expect_true(is.matrix(pkd[[1L]]))
    expect_equal(colnames(pkd[[1L]]), c("mz", "intensity"))
    pkd <- peaksData(sf, columns = peaksVariables(sf))
    expect_true(is.data.frame(pkd[[1L]]))
    expect_equal(colnames(pkd[[1L]]), c("mz", "intensity", "pk_ann"))
    expect_equal(pkd[[1L]][, "mz"], 1.1)
    expect_equal(pkd[[2L]][, "mz"], c(4.1, 5.1))
    expect_equal(pkd[[3L]][, "mz"], c(1.6, 1.7, 1.8))
    expect_equal(pkd[[1L]][, "pk_ann"], NA_character_)
    expect_equal(pkd[[2L]][, "pk_ann"], c("A", "B"))
    expect_equal(pkd[[3L]][, "pk_ann"], c("D", "E", "F"))

    ## spectraData
    spd <- spectraData(sf)
    expect_true(!any(colnames(spd) %in% peaksVariables(sf)))

    spd <- spectraData(sf, columns = c("rtime", "mz"))
    expect_equal(colnames(spd), c("rtime", "mz"))
    expect_equal(spectraData(sf, columns = "mz")$mz, sf$mz)
    spd <- spectraData(sf, columns = "intensity")
    expect_equal(colnames(spd), "intensity")
    expect_equal(spd$intensity, sf$intensity)
    spd <- spectraData(sf, columns = "pk_ann")
    expect_equal(
        spd$pk_ann, list(c(NA_character_), c("A", "B"), c("D", "E", "F")))
    spd <- spectraData(sf, columns = c("intensity", "pk_ann"))
    expect_equal(colnames(spd), c("intensity", "pk_ann"))
    expect_equal(
        spd$pk_ann, list(c(NA_character_), c("A", "B"), c("D", "E", "F")))

    ## check applyProcessing
    res <- applyProcessing(sf)
    expect_equal(res$rtime, sf$rtime)
    a <- NumericList(lapply(res$mz, unname), compress = FALSE)
    b <- NumericList(lapply(sf$mz, unname), compress = FALSE)
    expect_equal(a, b)
    a <- NumericList(lapply(res@backend$mz, unname), compress = FALSE)
    expect_equal(a, b)
    a <- NumericList(lapply(res$intensity, unname), compress = FALSE)
    b <- NumericList(lapply(sf$intensity, unname), compress = FALSE)
    expect_equal(a, b)
    a <- NumericList(lapply(res@backend$intensity, unname), compress = FALSE)
    expect_equal(a, b)
    expect_equal(res$pk_ann, sf$pk_ann)
    expect_equal(res@backend$pk_ann, sf$pk_ann)
    if (inherits(bcknd, "MsBackendMemory"))
        expect_true(is.matrix(res@backend@peaksData[[1L]]))
    expect_equal(
        res$pk_ann, list(c(NA_character_), c("A", "B"), c("D", "E", "F")))

    ## check replacement of peaks variables
    ## spectraData<-
    spd <- spectraData(sf, columns = union(spectraVariables(sf),
                                           peaksVariables(sf)))
    res <- sf
    expect_error(spectraData(res) <- spd, "non-empty processing queue")
    spd <- spectraData(sf)              # without peaks variables
    spectraData(res) <- spd
    expect_equal(rtime(res), rtime(sf))
    a <- NumericList(lapply(res$mz, unname), compress = FALSE)
    b <- NumericList(lapply(sf$mz, unname), compress = FALSE)
    expect_equal(a, b)
    a <- NumericList(lapply(res$intensity, unname), compress = FALSE)
    b <- NumericList(lapply(sf$intensity, unname), compress = FALSE)
    expect_equal(a, b)
    expect_equal(res$pk_ann, sf$pk_ann)
    expect_true(length(res@processingQueue) == 0)

    ## $<-
    res <- sf
    res$rtime <- 1:3
    expect_equal(res$rtime, 1:3)
    expect_error(res$pk_ann <- list(c(NA_character_), c("A", "D"),
                                    c("E", "F", "G")), "non-empty processing")
}, cases(
       MsBackendMemory = list(bcknd = MsBackendMemory()),
       MsBackendDataFrame = list(bcknd = MsBackendDataFrame())
   ))

test_that("combinePeaks,Spectra works", {
    x <- sps_dia[5:15]
    res <- combinePeaks(x, msLevel. = 1, tolerance = 0.1)
    expect_equal(peaksData(x[2]), peaksData(res[2]))
    expect_equal(lengths(x)[-5], lengths(res)[-5])
    expect_true(lengths(x)[5] > lengths(res)[5])

    res <- combinePeaks(x, tolerance = 0.1, intensityFun = median,
                        mzFun = median)
    expect_true(all(lengths(x) > lengths(res)))
    res_1 <- .peaks_combine(peaksData(x)[[1L]], tolerance = 0.1, ppm = 20,
                            intensityFun = median, mzFun = median,
                            msLevel = 1L, spectrumMsLevel = 1L)
    expect_equal(res_1, peaksData(res)[[1L]])
})

test_that("processingChunkSize works", {
    expect_equal(processingChunkSize(sps_dia), Inf)
    tmp <- sps_dia
    processingChunkSize(tmp) <- 10
    expect_equal(processingChunkSize(tmp), 10)
})

test_that("entropy,Spectra works", {
  sps <- Spectra()
  res <- entropy(sps)
  expect_identical(res, numeric())

  df <- DataFrame(msLevel = c(1L, 2L), centroided = TRUE)
  df$mz <- list(1:3, 1:3)
  df$intensity <- list(c(5, 9, 3), c(9, 8, 2))
  sps <- Spectra(df)

  res <- entropy(sps, normalized = TRUE)
  expect_identical(res, vapply(df$intensity, MsCoreUtils::nentropy, numeric(1)))

  res <- entropy(sps, normalized = FALSE)
  expect_identical(res, vapply(df$intensity, MsCoreUtils::entropy, numeric(1)))
})

test_that("dataStorageBasePath,dataStorageBasePath<-,MsBackendMzR works", {
    tmpd <- normalizePath(tempdir())
    file.copy(sciex_file, tmpd)

    tmp <- Spectra(sciex_mzr)
    expect_equal(dataStorageBasePath(tmp),
                 MsCoreUtils::common_path(sciex_file))
    tmp <- sciex_mzr
    tmp <- Spectra(tmp)
    dataStorageBasePath(tmp) <- tmpd
    expect_true(validObject(tmp@backend))
    bp <- normalizePath(dataStorageBasePath(tmp))
    expect_equal(bp, tmpd)

    #' errors
    expect_error(dataStorageBasePath(tmp) <- "some path", "Provided path")
})


test_that("asDataFrame works", {
    sciex_file <- normalizePath(
        dir(system.file("sciex", package = "msdata"), full.names = TRUE))
    sp <- Spectra(sciex_file)
    ## Full dataframe
    df <- asDataFrame(sp)
    expect_identical(nrow(df), sum(sapply(peaksData(sp), nrow)))
    expect_identical(ncol(df), length(spectraVariables(sp)) + 2L)
    expect_identical(names(df), c("mz", "intensity", spectraVariables(sp)))
    ## Three first scans and 2 spectra variables
    df <- asDataFrame(sp, i = 1:3, spectraVars = c("msLevel", "rtime"))
    expect_identical(nrow(df), sum(sapply(peaksData(sp[1:3]), nrow)))
    expect_identical(ncol(df), 2L + 2L)
    ## Three first scans and no spectra variables
    df <- asDataFrame(sp, i = 1:3, spectraVars = NULL)
    expect_identical(nrow(df), sum(sapply(peaksData(sp[1:3]), nrow)))
    expect_identical(ncol(df), 2L)
    expect_identical(names(df), c("mz", "intensity"))
})

test_that("estimatePrecursorIntensity works", {
    fls <- msdata::proteomics(full.names = TRUE)[c(5, 3)]
    second <- Spectra(fls[2], backend = MsBackendMzR())
    both <- Spectra(fls, backend = MsBackendMzR())

    res_second <- estimatePrecursorIntensity(second)
    res_both <- estimatePrecursorIntensity(both)
    expect_equal(res_second, res_both[510:length(res_both)])
})

test_that("precursorMz<-,Spectra works", {
    a <- sps_dda[1:3]
    precursorMz(a) <- c(12.3, 1.1, 34.3)
    expect_equal(precursorMz(a), c(12.3, 1.1, 34.3))
})

test_that("processingChunkSize works", {
    s <- Spectra()
    expect_equal(processingChunkSize(s), Inf)
    processingChunkSize(s) <- 1000
    expect_equal(processingChunkSize(s), 1000)
    expect_error(processingChunkSize(s) <- c(1, 2), "length 1")
    expect_error(processingChunkSize(s) <- "A", "character")
})

test_that("processingChunkFactor works", {
    s <- Spectra()
    expect_equal(processingChunkFactor(s), factor())
    tmp <- Spectra(sciex_mzr)

    expect_equal(length(processingChunkFactor(tmp)), length(tmp))
    expect_true(is.factor(processingChunkFactor(tmp)))

    processingChunkSize(tmp) <- 1000
    res <- processingChunkFactor(tmp)
    expect_true(is.factor(res))
    expect_true(length(res) == length(tmp))
    expect_equal(levels(res), c("1", "2"))

    expect_equal(.parallel_processing_factor(tmp), processingChunkFactor(tmp))

    expect_error(processingChunkFactor("a"), "signature")
})

test_that("applyProcessing works", {
    ## Initialize required objects.
    sps_mzr <- filterRt(Spectra(sciex_mzr), rt = c(10, 20))
    ## Add processings.
    centroided(sps_mzr) <- TRUE
    sps_mzr <- replaceIntensitiesBelow(sps_mzr, threshold = 5000,
                                       value = NA_real_)
    sps_mzr <- filterIntensity(sps_mzr)
    expect_true(length(sps_mzr@processingQueue) == 2)
    expect_error(applyProcessing(sps_mzr), "is read-only")

    ## Create writeable backends.
    sps_mem <- setBackend(sps_mzr, backend = MsBackendDataFrame())
    sps_h5 <- setBackend(sps_mzr, backend = MsBackendHdf5Peaks(),
                         files = c(tempfile(), tempfile()),
                         f = rep(1, length(sps_mzr)))
    expect_true(length(sps_mem@processingQueue) == 2)
    expect_true(length(sps_h5@processingQueue) == 2)
    expect_identical(peaksData(sps_mzr), peaksData(sps_mem))
    expect_identical(peaksData(sps_h5), peaksData(sps_mem))

    ## MsBackendDataFrame
    res <- applyProcessing(sps_mem)
    expect_true(length(res@processingQueue) == 0)
    expect_true(length(res@processing) > length(sps_mem@processing))
    expect_identical(rtime(res), rtime(sps_mem))
    expect_identical(peaksData(res), peaksData(sps_mem))

    ## MsBackendHdf5Peaks
    res <- applyProcessing(sps_h5)
    expect_true(length(res@processingQueue) == 0)
    expect_true(length(res@processing) > length(sps_h5@processing))
    expect_identical(rtime(res), rtime(sps_mem))
    expect_identical(peaksData(res), peaksData(sps_mem))
    expect_true(all(res@backend@modCount > sps_h5@backend@modCount))

    ## Applying the processing queue invalidated the original object!
    expect_error(peaksData(sps_h5))
    sps_h5 <- setBackend(sps_mzr, backend = MsBackendHdf5Peaks(),
                         files = c(tempfile(), tempfile()),
                         f = rep(1, length(sps_mzr)))

    ## Use an arbitrary splitting factor ensuring that the results are still OK.
    f <- rep(letters[1:9], 8)
    f <- sample(f)

    ## MsBackendHdf5Peaks
    res <- applyProcessing(sps_mem, f = f)
    expect_true(length(res@processingQueue) == 0)
    expect_true(length(res@processing) > length(sps_mem@processing))
    expect_identical(rtime(res), rtime(sps_mem))
    expect_identical(peaksData(res), peaksData(sps_mem))

    ## MsBackendHdf5Peaks: throws an error, because the factor f does not
    ## match the dataStorage.
    expect_error(applyProcessing(sps_h5, f = f))

    sps_h5 <- setBackend(sps_mzr, backend = MsBackendHdf5Peaks(),
                         files = c(tempfile(), tempfile()),
                         f = rep(1, length(sps_mzr)))
    res <- applyProcessing(sps_h5, f = rep(1, length(sps_h5)))
    expect_true(length(res@processingQueue) == 0)
    expect_true(length(res@processing) > length(sps_h5@processing))
    expect_identical(rtime(res), rtime(sps_mem))
    expect_identical(peaksData(res), peaksData(sps_mem))
    expect_true(all(res@backend@modCount > sps_h5@backend@modCount))

    expect_error(applyProcessing(sps_mem, f = 1:2), "has to be equal to the")
})

test_that("peaksData,Spectra works ", {
    a <- sps_dda[1:3]
    res <- peaksData(a)
    expect_s4_class(res, "SimpleList")
    res <- peaksData(a, return.type = "list")
    expect_true(is.list(res))
})

test_that("cbind2, Spectra works", {
    n <- length(sps_dda)
    df <- data.frame(new1 = rep(1, n), new2 = rep("a", n))
    tmp <- cbind2(sps_dda, df)
    expect_true(is(tmp, "Spectra"))
})

test_that("longForm,Spectra works", {
    a <- sps_dda[1:30]
    ## empty processing queue
    res <- longForm(a)
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c(spectraVariables(a), peaksVariables(a)))
    expect_equal(res$mz, unlist(mz(a)))
    expect_equal(res$intensity, unlist(intensity(a)))

    ## selected columns
    res <- longForm(a, c("rtime", "mz", "intensity"))
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c("rtime", "mz", "intensity"))
    expect_equal(res$mz, unlist(mz(a)))
    expect_equal(res$intensity, unlist(intensity(a)))

    ## no peaks variables
    res <- longForm(a, c("msLevel", "rtime"))
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c("msLevel", "rtime"))
    expect_equal(res$msLevel, msLevel(a))
    expect_equal(res$rtime, rtime(a))

    ## no spectra variables
    res <- longForm(a, c("mz", "intensity"))
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c("mz", "intensity"))
    expect_equal(res$mz, unlist(mz(a)))
    expect_equal(res$intensity, unlist(intensity(a)))

    ## non-existing column
    expect_error(longForm(a, c("rtime", "aaa"), "not available"))

    ## non-empty processing queue
    a <- filterIntensity(a, 80)
    res <- longForm(a)
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c(spectraVariables(a), peaksVariables(a)))
    expect_equal(res$mz, unname(unlist(mz(a))))
    expect_equal(res$intensity, unname(unlist(intensity(a))))
    expect_true(nrow(res) < length(a)) # because we have empty spectra

    ## selected columns
    res <- longForm(a, c("rtime", "mz", "intensity"))
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c("rtime", "mz", "intensity"))
    expect_equal(res$mz, unname(unlist(mz(a))))
    expect_equal(res$intensity, unname(unlist(intensity(a))))

    ## no peaks variables
    res <- longForm(a, c("msLevel", "rtime"))
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c("msLevel", "rtime"))
    expect_equal(res$msLevel, msLevel(a))
    expect_equal(res$rtime, rtime(a))

    ## no spectra variables
    res <- longForm(a, c("mz", "intensity"))
    expect_true(is.data.frame(res))
    expect_equal(colnames(res), c("mz", "intensity"))
    expect_equal(res$mz, unname(unlist(mz(a))))
    expect_equal(res$intensity, unname(unlist(intensity(a))))

    ## non-existing column
    expect_error(longForm(a, c("rtime", "aaa"), "not available"))
})

test_that("spectraVariableMapping,Spectra works", {
    df <- DataFrame(msLevel = c(1L, 2L))
    s <- Spectra(df)
    expect_error(spectraVariableMapping(s), "not supported")
    expect_error(spectraVariableMapping(s) <- 323, "not supported")
})
