logitboost <- function(xlearn, ylearn, xtest, mfinal, presel = 0,
                       estimate = 0, verbose = FALSE)
  {
    ## The binary version, always used if called from crossval()
    if (nlevels(as.factor(ylearn))==2)
      {      
        ## Feature Preselection
        if (presel > 0)
          {
            s       <- apply(xlearn, 2, score, ylearn)
            quality <- apply(rbind(s,-s+(sum(ylearn==0)*sum(ylearn==1))),2,max)
            genes   <- rev(order(quality))[1:presel]
            xlearn  <- xlearn[, genes]
            xtest   <- xtest[ , genes, drop = FALSE]
          }

        ## Estimation of the stopping parameter
        if (estimate>0)
          {
            if (verbose) {print("Stopping Parameter Estimation")}
            likeli       <- numeric(mfinal)
            probabs      <- crossval(xlearn, ylearn, estimate, mfinal)$probs
            for (k in 1:mfinal)
              {
                a          <- pmax(log(probabs[,k]),   -1e36)
                b          <- pmax(log(1-probabs[,k]), -1e36)
                likeli[k]  <- (ylearn%*%a)+((1-ylearn)%*%b)
              }
          }

        ## Length of training and test data
        learn    <- dim(xlearn)[1]         
        test     <- dim(xtest)[1]

        ## Initialization
        Flearn   <- numeric(learn)             
        Ftest    <- numeric(test)              
        flearn   <- numeric(learn)             
        ftest    <- numeric(test)              
        z        <- numeric(learn)             
        w        <- numeric(learn)             
        plearn   <- rep(1/2, learn)
        ptest    <- matrix(0, test, mfinal)

        ## Boosting Iterations
        if (verbose) { print("Boosting Iterations") }
        for (m in 1:mfinal)
          {
            ## Computation of working response and weights
            w      <- pmax(plearn*(1-plearn), 1e-24)
            z      <- (ylearn-plearn)/w

            ## Setting the arguments for rpart
            cntrl  <- rpart.control(maxdepth=1, minsplit=learn-1, maxcompete=0,
                                    maxsurrogate=0, cp=0, xval=0)

            ## Fitting the tree
            xx     <- xlearn
            fit    <- rpart(z~xx, weights = w/mean(w), control = cntrl)
            flearn <- predict(fit)
            xx     <- xtest
            ftest  <- predict(fit, newdata = data.frame(xx))

            ## Updating and probabilities
            Flearn    <- Flearn + (1/2)*flearn
            Ftest     <- Ftest  + (1/2)*ftest
            plearn    <- 1/(1+exp((-2)*Flearn))
            ptest[,m] <- 1/(1+exp((-2)*Ftest))
          }

        ## Output
        output   <- list(probs = ptest)
        if (estimate>0){output <- list(probs = ptest,loglikeli = matrix(likeli,
                                       nr = test, nc = mfinal, byrow = TRUE))}
      }

    ## The multiclass version, only used if logitboost() is called directly
    if (nlevels(as.factor(ylearn))>2)
      {
        ## Preliminaries
        K      <- nlevels(as.factor(ylearn))
        likeli <- array(0, c(dim(xtest)[1], mfinal, K))
        ptest  <- array(0, c(dim(xtest)[1], mfinal, K))

        ## Looping over the K classes
        for (k in 0:(K-1))
          {
            ## Defining the response
            yyl <- as.numeric(ylearn==k)
            
            ## Feature Preselection
            if (presel > 0)
              {
                s       <- apply(xlearn, 2, score, yyl)
                quality <- apply(rbind(s,-s+(sum(yyl==0)*sum(yyl==1))),2,max)
                genes   <- rev(order(quality))[1:presel]
                xxl     <- xlearn[, genes]
                xxt     <- xtest[ , genes, drop = FALSE]
              }
            else {  ## VC guesswork -- dies with no xxl found if no presel
                s <- apply(xlearn, 2, score, yyl)
                quality <- apply(rbind(s, -s + (sum(yyl == 0) *
                  sum(yyl == 1))), 2, max)
                genes <- rev(order(quality))[1:length(quality)]
                xxl <- xlearn[, genes]
                xxt <- xtest[, genes, drop = FALSE]
            }


            ## Estimation of the stopping parameter
            if (estimate>0)
              {
                if (verbose) {print("Stopping Parameter Estimation")}
                probabs      <- crossval(xxl, yyl, estimate, mfinal)$probs
                for (i in 1:mfinal)
                  {
                    a                 <- pmax(log(probabs[,i]),   -1e36)
                    b                 <- pmax(log(1-probabs[,i]), -1e36)
                    for (q in 1:dim(xtest)[1])
                      {
                        likeli[q,i,(k+1)] <- (yyl%*%a)+((1-yyl)%*%b)
                      }
                  }
              }

            ## Length of training and test data
            learn    <- dim(xxl)[1]         
            test     <- dim(xxt)[1]

            ## Initialization
            Flearn   <- numeric(learn)             
            Ftest    <- numeric(test)              
            flearn   <- numeric(learn)             
            ftest    <- numeric(test)              
            z        <- numeric(learn)             
            w        <- numeric(learn)             
            plearn   <- rep(1/2, learn)

            ## Boosting Iterations
            if (verbose) { print("Boosting Iterations") }
            for (m in 1:mfinal)
              {
                ## Computation of working response and weights
                w      <- pmax(plearn*(1-plearn), 1e-24)
                z      <- (yyl-plearn)/w

                ## Setting the arguments for rpart
                cntrl  <- rpart.control(maxdepth=1, minsplit=learn-1, xval=0,
                                        maxcompete=0, cp=0, maxsurrogate=0, )

                ## Fitting the tree
                xx     <- xxl
                fit    <- rpart(z~xx, weights = w/mean(w), control = cntrl)
                flearn <- predict(fit)
                xx     <- xxt
                ftest  <- predict(fit, newdata = data.frame(xx))

                ## Updating and probabilities
                Flearn          <- Flearn + (1/2)*flearn
                Ftest           <- Ftest  + (1/2)*ftest
                plearn          <- 1/(1+exp((-2)*Flearn))
                ptest[,m,(k+1)] <- 1/(1+exp((-2)*Ftest))
              }
          }
             
        ## Output
        output   <- list(probs = ptest)
        if (estimate>0){output <- list(probs=ptest,loglikeli=likeli)}
      }
    
    ## Final output
    class(output) <- "logitboost" # VJC
    attr(output,"call") <- match.call() #VJC
    output
  }





