#' @title KallistoMetadata S4 class
#' 
#' @description An S4 class that is the desendant of the AbundanceMetadata class. 
#' It contains all metadata needed to run kallisto analysis.
#' All slots of this class have a default value. You do not need to edit them to 
#' run the package
#'
#' @slot download_kallisto A logical allowing to use an already installed version 
#' of kallisto or to download a version that will be used only by this package
#' @slot kallisto_windows_url URL to the binary of kallisto for windows
#' @slot kallisto_linux_url URL to the binary of kallisto for linux
#' @slot kallisto_osx_url URL to the binary of kallisto for MacOS
#' @slot kallisto_windows_dir Name of the directory where kallisto will be installed
#' on windows
#' @slot kallisto_linux_dir Name of the directory where kallisto will be installed
#' on linux
#' @slot kallisto_osx_dir Name of the directory where kallisto will be installed
#' on Mac
#' @slot unix_kallisto_name Name of the kallisto executable in linux and macOS
#' @slot windows_kallisto_name Name of the kallisto executable in windows
#' @slot index_file Name of index file generated by kallisto with default kmer 
#' size. It will be generated using the fasta file that contains both 
#' transcriptomic and intergenic regions. Do not use an index you generated 
#' outside of this package. This file is created by the pipeline. 
#' You should edit this slot only if you already have such a file with a 
#' different name. This file must be store at get_tool_path()
#' @slot k15_index_file same as index_file. This index is generated with smallest 
#' kmers and will be used only for libraries containing reads smallest than 50nt.
#' @slot single_end_parameters kallisto parameters used to run a single end mapping
#' @slot pair_end_parameters kallisto parameters used to run a pair end mapping
#' @slot overwrite_index logical allowing to overwrite already existing index. FALSE by
#' default. Then by default already existing index files will not be generated again.
#' @slot overwrite_quant logical allowing to overwrite already existing abundance.txt 
#' files. FALSE by default. Then by default already existing quantitfdication files will 
#' not be generated again.
#' @slot overwrite_calls logical allowing to overwrite already existing present/absent 
#' calls. FALSE by default. Then by default already generated calls will not be 
#' generated again.
#' 
#' @exportClass KallistoMetadata

KallistoMetadata <- setClass(
    # Set the name for the class
    Class = "KallistoMetadata",
    
    # Define the slots
    representation = representation(
        download_kallisto = "logical",
        kallisto_windows_url = "character",
        kallisto_linux_url = "character",
        kallisto_osx_url = "character",
        kallisto_windows_dir = "character",
        kallisto_linux_dir = "character",
        kallisto_osx_dir = "character",
        unix_kallisto_name = "character",
        windows_kallisto_name = "character",
        index_file = "character",
        k15_index_file = "character",
        single_end_parameters = "character",
        pair_end_parameters = "character",
        overwrite_index = "logical",
        overwrite_quant = "logical",
        overwrite_calls = "logical"
    ),
    
    # Set the default values for the slots.
    prototype = prototype(
        # slots defined in parent AbundanceMetadata Class
        tool_name = "kallisto",
        abundance_file = "abundance.tsv",
        abundance_file_without_tx_version = "abundance_without_tx_version.tsv",
        transcript_id_header = "target_id",
        count_header = "est_counts",
        abundance_header = "tpm",
        eff_length_header = "eff_length",
        read_size_kmer_threshold = 60,
        # slots specific to Class
        download_kallisto = FALSE,
        kallisto_windows_url = 
            "https://github.com/pachterlab/kallisto/releases/download/v0.45.0/kallisto_windows-v0.45.0.zip",
        kallisto_linux_url = 
            "https://github.com/pachterlab/kallisto/releases/download/v0.45.0/kallisto_linux-v0.45.0.tar.gz",
        kallisto_osx_url = 
            "https://github.com/pachterlab/kallisto/releases/download/v0.45.0/kallisto_mac-v0.45.0.tar.gz",
        kallisto_windows_dir = "kallisto",
        kallisto_linux_dir = "kallisto_linux-v0.45.0",
        kallisto_osx_dir = "kallisto",
        unix_kallisto_name = "kallisto",
        windows_kallisto_name = "kallisto.exe",
        index_file = "transcriptome.idx",
        #transcriptome index file for kmer size = 15
        k15_index_file = "transcriptome_k15.idx", 
        single_end_parameters = "-t 1 --single -l 180 -s 30",
        pair_end_parameters = "-t 1",
        overwrite_index = FALSE,
        overwrite_quant = FALSE,
        overwrite_calls = TRUE
    ),
    
    # This class also have attributs/methods of AbundanceMetadata class (inheritence)
    contains = "AbundanceMetadata"
)
