context('extractSignalRawData()')

skip_if_not_installed('faahKO',  minimum_version = '1.18.0')
library(faahKO)


## Input and expected data
# use ko15.CDf file from the pkg faahKO
raw_data <- MSnbase::readMSData( system.file('cdf/KO/ko15.CDF', package = "faahKO"), centroided=TRUE, mode='onDisk')

test_that('1 numeric rt-mz window', {
  # Expected signal
  expected_rt  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  expected_signal   <- list(data.frame(rt=expected_rt, mz=expected_mz, int=expected_int))
  expected_messages <- 'Reading data from 1 windows\n'

  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=c(3310., 3390.), mz=c(520., 522.4), msLevel=1L, verbose=TRUE))

  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages (skip timing)
  expect_equal(length(result_extractSignalRawData$messages), 2)
  expect_equal(result_extractSignalRawData$messages[1], expected_messages)
})

test_that('2 matrix rt-mz windows', {
  # Expected signal
  expected_rt1  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz1  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int1 <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
 
  expected_rt2  <- c(2900.441, 2900.441, 2900.441, 2902.006, 2902.006, 2902.006, 2903.571, 2903.571, 2903.571, 2905.136, 2905.136, 2905.136, 2906.701, 2906.701, 2906.701, 2906.701, 2908.266, 2908.266, 2908.266, 2908.266, 2909.831, 2909.831, 2909.831, 2911.396, 2911.396, 2911.396, 2912.961, 2912.961, 2912.961, 2914.526, 2914.526, 2914.526, 2916.091, 2916.091, 2916.091, 2917.656, 2917.656, 2917.656, 2919.221, 2919.221, 2919.221, 2920.786, 2920.786, 2920.786, 2922.351, 2922.351, 2922.351, 2923.916, 2923.916, 2923.916, 2923.916, 2925.48, 2925.48, 2925.48, 2927.045, 2927.045, 2927.045, 2928.61, 2928.61, 2928.61, 2930.175, 2930.175, 2930.175, 2931.74, 2931.74, 2931.74, 2933.305, 2933.305, 2933.305, 2934.87, 2934.87, 2934.87, 2936.435, 2936.435, 2936.435, 2938, 2938, 2938, 2939.565, 2939.565, 2939.565, 2941.13, 2941.13, 2941.13, 2941.13, 2942.695, 2942.695, 2942.695, 2944.26, 2944.26, 2944.26, 2945.825, 2945.825, 2945.825, 2947.39, 2947.39, 2947.39, 2948.955, 2948.955, 2948.955, 2950.52, 2950.52, 2950.52, 2950.52, 2952.085, 2952.085, 2952.085, 2953.65, 2953.65, 2953.65, 2953.65, 2955.215, 2955.215, 2955.215, 2955.215, 2956.78, 2956.78, 2956.78, 2958.344, 2958.344, 2958.344, 2959.909, 2959.909, 2959.909, 2959.909, 2961.474, 2961.474, 2961.474, 2963.039, 2963.039, 2963.039, 2963.039, 2964.604, 2964.604, 2964.604, 2964.604, 2966.169, 2966.169, 2966.169, 2966.169, 2967.734, 2967.734, 2967.734, 2969.299, 2969.299, 2969.299, 2970.864, 2970.864, 2970.864, 2972.429, 2972.429, 2972.429, 2973.994, 2973.994, 2973.994, 2975.559, 2975.559, 2975.559, 2977.124, 2977.124, 2977.124, 2978.689, 2978.689, 2978.689, 2980.254, 2980.254, 2980.254, 2981.819, 2981.819, 2981.819, 2983.384, 2983.384, 2983.384, 2984.949, 2984.949, 2984.949, 2986.514, 2986.514, 2986.514, 2986.514, 2986.514, 2988.079, 2988.079, 2988.079, 2988.079, 2989.644, 2989.644, 2989.644)
  expected_mz2  <- c(460.10000610351562, 461.20001220703125, 462.30001831054688, 460.10000610351562, 461.20001220703125, 462.30001831054688, 460.10000610351562, 461.30001831054688, 462.20001220703125, 460.10000610351562, 461.30001831054688, 462.20001220703125, 460.10000610351562, 460.39999389648438, 461.39999389648438, 462.10000610351562, 460.00000000000000, 460.39999389648438, 461.39999389648438, 462.10000610351562, 460.39999389648438, 461.39999389648438, 462.10000610351562, 460.89999389648438, 461.30001831054688, 462.20001220703125, 460.00000000000000, 460.89999389648438, 462.20001220703125, 460.00000000000000, 460.89999389648438, 462.20001220703125, 460.00000000000000, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.00000000000000, 460.10000610351562, 461.00000000000000, 461.89999389648438, 460.20001220703125, 460.80001831054688, 462.00000000000000, 460.80001831054688, 461.30001831054688, 462.10000610351562, 460.39999389648438, 460.80001831054688, 461.30001831054688, 462.20001220703125, 460.39999389648438, 461.30001831054688, 462.30001831054688, 460.00000000000000, 461.20001220703125, 462.20001220703125, 460.00000000000000, 460.89999389648438, 462.10000610351562, 460.00000000000000, 460.89999389648438, 462.00000000000000, 460.10000610351562, 460.89999389648438, 461.89999389648438, 460.10000610351562, 461.00000000000000, 461.89999389648438, 460.20001220703125, 461.00000000000000, 462.00000000000000, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.39999389648438, 461.20001220703125, 462.10000610351562, 460.30001831054688, 460.50000000000000, 461.20001220703125, 462.20001220703125, 460.30001831054688, 461.10000610351562, 462.30001831054688, 460.30001831054688, 461.10000610351562, 462.30001831054688, 460.30001831054688, 461.10000610351562, 462.20001220703125, 460.30001831054688, 461.10000610351562, 462.20001220703125, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.00000000000000, 460.39999389648438, 461.10000610351562, 462.10000610351562, 460.39999389648438, 461.30001831054688, 462.10000610351562, 460.39999389648438, 461.00000000000000, 461.39999389648438, 462.20001220703125, 460.39999389648438, 461.00000000000000, 461.30001831054688, 462.20001220703125, 460.39999389648438, 461.00000000000000, 462.30001831054688, 460.39999389648438, 461.00000000000000, 462.30001831054688, 460.00000000000000, 460.39999389648438, 461.00000000000000, 462.20001220703125, 460.10000610351562, 461.10000610351562, 462.30001831054688, 460.20001220703125, 461.10000610351562, 461.80001831054688, 462.30001831054688, 460.30001831054688, 461.00000000000000, 461.80001831054688, 462.30001831054688, 460.30001831054688, 461.00000000000000, 461.80001831054688, 462.30001831054688, 460.20001220703125, 461.10000610351562, 462.30001831054688, 460.10000610351562, 461.20001220703125, 462.30001831054688, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.20001220703125, 461.80001831054688, 462.20001220703125, 460.30001831054688, 461.30001831054688, 461.80001831054688, 462.20001220703125, 460.39999389648438, 461.30001831054688, 462.10000610351562)
  expected_int2 <- c(840, 1039, 818, 825, 1026, 728, 709, 975, 748, 563, 885, 763, 427, 410, 854, 713, 414, 396, 876, 585, 343, 806, 508, 731, 709, 562, 784, 929, 584, 793, 1055, 561, 648, 1164, 622, 533, 1287, 824, 443, 1297, 971, 468, 1275, 954, 1156, 973, 905, 825, 924, 1140, 879, 821, 1263, 806, 804, 1254, 647, 829, 1403, 630, 763, 1471, 685, 772, 1487, 723, 817, 1623, 697, 858, 1721, 705, 929, 1586, 818, 998, 1537, 908, 989, 1673, 844, 962, 968, 1621, 764, 968, 1371, 758, 931, 1193, 692, 831, 1033, 699, 765, 893, 823, 741, 835, 943, 665, 691, 785, 1021, 635, 774, 979, 705, 532, 703, 858, 865, 613, 518, 698, 932, 813, 621, 785, 884, 628, 500, 537, 806, 806, 403, 715, 952, 347, 755, 660, 1041, 364, 782, 632, 1051, 380, 783, 624, 1043, 387, 730, 1101, 487, 905, 1206, 641, 1081, 1313, 769, 1151, 1243, 723, 1065, 1143, 589, 914, 1030, 539, 819, 965, 608, 795, 937, 679, 866, 970, 784, 977, 1059, 795, 1041, 1061, 794, 1021, 1166, 621, 698, 905, 682, 1209, 495, 947, 600, 1113, 419, 1067, 932)
  
  expected_signal   <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))

  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3310., 3390.,2900., 2990.),ncol=2, byrow=TRUE), mz=matrix(c(520., 522.4, 460., 462.4), ncol=2, byrow=TRUE), msLevel=1L, verbose=FALSE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(length(result_extractSignalRawData$messages), 0)
})

test_that('2 data.frame rt-mz windows', {
  # Expected signal
  expected_rt1  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz1  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int1 <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  
  expected_rt2  <- c(2900.441, 2900.441, 2900.441, 2902.006, 2902.006, 2902.006, 2903.571, 2903.571, 2903.571, 2905.136, 2905.136, 2905.136, 2906.701, 2906.701, 2906.701, 2906.701, 2908.266, 2908.266, 2908.266, 2908.266, 2909.831, 2909.831, 2909.831, 2911.396, 2911.396, 2911.396, 2912.961, 2912.961, 2912.961, 2914.526, 2914.526, 2914.526, 2916.091, 2916.091, 2916.091, 2917.656, 2917.656, 2917.656, 2919.221, 2919.221, 2919.221, 2920.786, 2920.786, 2920.786, 2922.351, 2922.351, 2922.351, 2923.916, 2923.916, 2923.916, 2923.916, 2925.48, 2925.48, 2925.48, 2927.045, 2927.045, 2927.045, 2928.61, 2928.61, 2928.61, 2930.175, 2930.175, 2930.175, 2931.74, 2931.74, 2931.74, 2933.305, 2933.305, 2933.305, 2934.87, 2934.87, 2934.87, 2936.435, 2936.435, 2936.435, 2938, 2938, 2938, 2939.565, 2939.565, 2939.565, 2941.13, 2941.13, 2941.13, 2941.13, 2942.695, 2942.695, 2942.695, 2944.26, 2944.26, 2944.26, 2945.825, 2945.825, 2945.825, 2947.39, 2947.39, 2947.39, 2948.955, 2948.955, 2948.955, 2950.52, 2950.52, 2950.52, 2950.52, 2952.085, 2952.085, 2952.085, 2953.65, 2953.65, 2953.65, 2953.65, 2955.215, 2955.215, 2955.215, 2955.215, 2956.78, 2956.78, 2956.78, 2958.344, 2958.344, 2958.344, 2959.909, 2959.909, 2959.909, 2959.909, 2961.474, 2961.474, 2961.474, 2963.039, 2963.039, 2963.039, 2963.039, 2964.604, 2964.604, 2964.604, 2964.604, 2966.169, 2966.169, 2966.169, 2966.169, 2967.734, 2967.734, 2967.734, 2969.299, 2969.299, 2969.299, 2970.864, 2970.864, 2970.864, 2972.429, 2972.429, 2972.429, 2973.994, 2973.994, 2973.994, 2975.559, 2975.559, 2975.559, 2977.124, 2977.124, 2977.124, 2978.689, 2978.689, 2978.689, 2980.254, 2980.254, 2980.254, 2981.819, 2981.819, 2981.819, 2983.384, 2983.384, 2983.384, 2984.949, 2984.949, 2984.949, 2986.514, 2986.514, 2986.514, 2986.514, 2986.514, 2988.079, 2988.079, 2988.079, 2988.079, 2989.644, 2989.644, 2989.644)
  expected_mz2  <- c(460.10000610351562, 461.20001220703125, 462.30001831054688, 460.10000610351562, 461.20001220703125, 462.30001831054688, 460.10000610351562, 461.30001831054688, 462.20001220703125, 460.10000610351562, 461.30001831054688, 462.20001220703125, 460.10000610351562, 460.39999389648438, 461.39999389648438, 462.10000610351562, 460.00000000000000, 460.39999389648438, 461.39999389648438, 462.10000610351562, 460.39999389648438, 461.39999389648438, 462.10000610351562, 460.89999389648438, 461.30001831054688, 462.20001220703125, 460.00000000000000, 460.89999389648438, 462.20001220703125, 460.00000000000000, 460.89999389648438, 462.20001220703125, 460.00000000000000, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.00000000000000, 460.10000610351562, 461.00000000000000, 461.89999389648438, 460.20001220703125, 460.80001831054688, 462.00000000000000, 460.80001831054688, 461.30001831054688, 462.10000610351562, 460.39999389648438, 460.80001831054688, 461.30001831054688, 462.20001220703125, 460.39999389648438, 461.30001831054688, 462.30001831054688, 460.00000000000000, 461.20001220703125, 462.20001220703125, 460.00000000000000, 460.89999389648438, 462.10000610351562, 460.00000000000000, 460.89999389648438, 462.00000000000000, 460.10000610351562, 460.89999389648438, 461.89999389648438, 460.10000610351562, 461.00000000000000, 461.89999389648438, 460.20001220703125, 461.00000000000000, 462.00000000000000, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.39999389648438, 461.20001220703125, 462.10000610351562, 460.30001831054688, 460.50000000000000, 461.20001220703125, 462.20001220703125, 460.30001831054688, 461.10000610351562, 462.30001831054688, 460.30001831054688, 461.10000610351562, 462.30001831054688, 460.30001831054688, 461.10000610351562, 462.20001220703125, 460.30001831054688, 461.10000610351562, 462.20001220703125, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.00000000000000, 460.39999389648438, 461.10000610351562, 462.10000610351562, 460.39999389648438, 461.30001831054688, 462.10000610351562, 460.39999389648438, 461.00000000000000, 461.39999389648438, 462.20001220703125, 460.39999389648438, 461.00000000000000, 461.30001831054688, 462.20001220703125, 460.39999389648438, 461.00000000000000, 462.30001831054688, 460.39999389648438, 461.00000000000000, 462.30001831054688, 460.00000000000000, 460.39999389648438, 461.00000000000000, 462.20001220703125, 460.10000610351562, 461.10000610351562, 462.30001831054688, 460.20001220703125, 461.10000610351562, 461.80001831054688, 462.30001831054688, 460.30001831054688, 461.00000000000000, 461.80001831054688, 462.30001831054688, 460.30001831054688, 461.00000000000000, 461.80001831054688, 462.30001831054688, 460.20001220703125, 461.10000610351562, 462.30001831054688, 460.10000610351562, 461.20001220703125, 462.30001831054688, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.20001220703125, 461.80001831054688, 462.20001220703125, 460.30001831054688, 461.30001831054688, 461.80001831054688, 462.20001220703125, 460.39999389648438, 461.30001831054688, 462.10000610351562)
  expected_int2 <- c(840, 1039, 818, 825, 1026, 728, 709, 975, 748, 563, 885, 763, 427, 410, 854, 713, 414, 396, 876, 585, 343, 806, 508, 731, 709, 562, 784, 929, 584, 793, 1055, 561, 648, 1164, 622, 533, 1287, 824, 443, 1297, 971, 468, 1275, 954, 1156, 973, 905, 825, 924, 1140, 879, 821, 1263, 806, 804, 1254, 647, 829, 1403, 630, 763, 1471, 685, 772, 1487, 723, 817, 1623, 697, 858, 1721, 705, 929, 1586, 818, 998, 1537, 908, 989, 1673, 844, 962, 968, 1621, 764, 968, 1371, 758, 931, 1193, 692, 831, 1033, 699, 765, 893, 823, 741, 835, 943, 665, 691, 785, 1021, 635, 774, 979, 705, 532, 703, 858, 865, 613, 518, 698, 932, 813, 621, 785, 884, 628, 500, 537, 806, 806, 403, 715, 952, 347, 755, 660, 1041, 364, 782, 632, 1051, 380, 783, 624, 1043, 387, 730, 1101, 487, 905, 1206, 641, 1081, 1313, 769, 1151, 1243, 723, 1065, 1143, 589, 914, 1030, 539, 819, 965, 608, 795, 937, 679, 866, 970, 784, 977, 1059, 795, 1041, 1061, 794, 1021, 1166, 621, 698, 905, 682, 1209, 495, 947, 600, 1113, 419, 1067, 932)
  
  expected_signal   <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=data.frame(rtMin=c(3310., 2900.), rtMax=c(3390., 2990.)), mz=data.frame(mzMin=c(520.,460.), mzMax=c(522.4, 462.4)), msLevel=1L, verbose=FALSE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(length(result_extractSignalRawData$messages), 0)
  })

test_that('1 numeric rt, 2 matrix mz (extend rt, no verbose)', {
  # Expected signal
  expected_rt1  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz1  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int1 <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  
  expected_rt2  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3330.803, 3332.368, 3332.368, 3332.368, 3333.933, 3333.933, 3333.933, 3335.498, 3335.498, 3335.498, 3337.063, 3337.063, 3337.063, 3337.063, 3338.628, 3338.628, 3338.628, 3340.193, 3340.193, 3340.193, 3341.758, 3341.758, 3341.758, 3343.323, 3343.323, 3343.323, 3344.888, 3344.888, 3344.888, 3346.453, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3354.278, 3355.843, 3355.843, 3355.843, 3357.408, 3357.408, 3357.408, 3358.973, 3358.973, 3358.973, 3360.538, 3360.538, 3360.538, 3362.102, 3362.102, 3362.102, 3363.667, 3363.667, 3363.667, 3365.232, 3365.232, 3365.232, 3366.797, 3366.797, 3366.797, 3368.362, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3379.317, 3379.317, 3380.882, 3380.882, 3380.882, 3380.882, 3382.447, 3382.447, 3382.447, 3384.012, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz2  <- c(460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.30001831054688, 462.20001220703125, 460.30001831054688, 461.30001831054688, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.10000610351562, 460.89999389648438, 461.30001831054688, 462.20001220703125, 460.10000610351562, 460.89999389648438, 461.30001831054688, 462.20001220703125, 460.20001220703125, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.10000610351562, 461.20001220703125, 462.20001220703125, 460.00000000000000, 461.10000610351562, 462.30001831054688, 460.00000000000000, 461.00000000000000, 462.30001831054688, 460.00000000000000, 460.30001831054688, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.39999389648438, 461.10000610351562, 462.20001220703125, 460.39999389648438, 461.10000610351562, 462.20001220703125, 460.00000000000000, 460.39999389648438, 461.10000610351562, 462.10000610351562, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.20001220703125, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.10000610351562, 460.10000610351562, 461.20001220703125, 462.10000610351562, 460.10000610351562, 461.20001220703125, 462.00000000000000, 460.00000000000000, 461.20001220703125, 462.00000000000000, 462.39999389648438, 460.00000000000000, 461.00000000000000, 462.00000000000000, 462.39999389648438, 460.00000000000000, 461.00000000000000, 462.10000610351562, 462.30001831054688, 460.00000000000000, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.30001831054688, 462.30001831054688, 460.10000610351562, 461.30001831054688, 462.30001831054688)
  expected_int2 <- c(308, 452, 2757, 250, 242, 499, 3260, 311, 547, 3423, 292, 356, 552, 3396, 372, 372, 547, 3410, 427, 536, 3420, 428, 653, 3286, 439, 694, 3117, 467, 582, 2929, 544, 438, 430, 2608, 636, 336, 336, 2204, 722, 1928, 738, 308, 1768, 613, 334, 1561, 483, 330, 1282, 389, 288, 1071, 343, 260, 964, 266, 229, 290, 952, 280, 372, 991, 333, 376, 939, 342, 293, 799, 339, 225, 717, 357, 235, 809, 371, 241, 961, 277, 372, 229, 1084, 393, 276, 1042, 449, 355, 953, 527, 432, 944, 537, 476, 1038, 476, 561, 1206, 418, 649, 1321, 421, 705, 1422, 517, 697, 1551, 646, 646, 1765, 705, 603, 1923, 629, 618, 2032, 592, 643, 1956, 579, 637, 1763, 544, 554, 1578, 445, 427, 1427, 422, 359, 1399, 481, 409, 1307, 543, 494, 1119, 479, 463, 982, 807, 444, 380, 870, 812, 493, 362, 857, 828, 592, 364, 923, 644, 394, 914, 616, 385, 843, 518, 379, 798, 383, 467, 790)
  
  expected_signal   <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=c(3310., 3390.), mz=matrix(c(520., 522.4, 460., 462.4), ncol=2, byrow=TRUE), msLevel=1L, verbose=FALSE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(length(result_extractSignalRawData$messages), 0)
})

test_that('2 matrix rt, 1 numeric mz (extend mz, no verbose)', {
  # Expected signal
  expected_rt1  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz1  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int1 <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  
  expected_rt2  <- c(2900.441, 2900.441, 2900.441, 2902.006, 2902.006, 2902.006, 2903.571, 2903.571, 2903.571, 2905.136, 2905.136, 2905.136, 2906.701, 2906.701, 2906.701, 2908.266, 2908.266, 2908.266, 2909.831, 2909.831, 2909.831, 2911.396, 2911.396, 2911.396, 2912.961, 2912.961, 2912.961, 2914.526, 2914.526, 2914.526, 2916.091, 2916.091, 2916.091, 2917.656, 2917.656, 2917.656, 2919.221, 2919.221, 2919.221, 2920.786, 2920.786, 2920.786, 2922.351, 2922.351, 2922.351, 2923.916, 2923.916, 2923.916, 2925.48, 2925.48, 2925.48, 2927.045, 2927.045, 2927.045, 2928.61, 2928.61, 2928.61, 2930.175, 2930.175, 2930.175, 2931.74, 2931.74, 2931.74, 2933.305, 2933.305, 2933.305, 2934.87, 2934.87, 2936.435, 2936.435, 2936.435, 2938, 2938, 2938, 2938, 2939.565, 2939.565, 2939.565, 2941.13, 2941.13, 2942.695, 2942.695, 2942.695, 2944.26, 2944.26, 2944.26, 2945.825, 2945.825, 2945.825, 2947.39, 2947.39, 2947.39, 2948.955, 2948.955, 2948.955, 2948.955, 2950.52, 2950.52, 2950.52, 2950.52, 2952.085, 2952.085, 2952.085, 2952.085, 2953.65, 2953.65, 2953.65, 2953.65, 2955.215, 2955.215, 2955.215, 2956.78, 2956.78, 2956.78, 2958.344, 2958.344, 2958.344, 2959.909, 2959.909, 2959.909, 2961.474, 2961.474, 2961.474, 2961.474, 2963.039, 2963.039, 2963.039, 2963.039, 2964.604, 2964.604, 2964.604, 2966.169, 2966.169, 2966.169, 2967.734, 2967.734, 2967.734, 2969.299, 2969.299, 2969.299, 2970.864, 2970.864, 2970.864, 2972.429, 2972.429, 2972.429, 2973.994, 2973.994, 2973.994, 2975.559, 2975.559, 2975.559, 2977.124, 2977.124, 2977.124, 2978.689, 2978.689, 2978.689, 2978.689, 2980.254, 2981.819, 2981.819, 2983.384, 2983.384, 2983.384, 2984.949, 2984.949, 2984.949, 2986.514, 2986.514, 2986.514, 2986.514, 2988.079, 2988.079, 2988.079, 2988.079, 2989.644, 2989.644, 2989.644)
  expected_mz2  <- c(520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.10003662109375, 522.10003662109375, 520.20001220703125, 521.10003662109375, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.20001220703125, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.20001220703125, 520.20001220703125, 521.10003662109375, 522.20001220703125, 520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.00000000000000, 522.00000000000000, 520.20001220703125, 521.10003662109375, 522.00000000000000, 520.10003662109375, 521.20001220703125, 522.20001220703125, 520.10003662109375, 521.10003662109375, 522.20001220703125, 520.10003662109375, 521.10003662109375, 522.00000000000000, 520.10003662109375, 521.10003662109375, 522.00000000000000, 520.10003662109375, 521.10003662109375, 521.90002441406250, 520.10003662109375, 521.00000000000000, 521.90002441406250, 520.20001220703125, 520.90002441406250, 521.90002441406250, 520.20001220703125, 520.90002441406250, 521.90002441406250, 520.20001220703125, 520.90002441406250, 520.20001220703125, 520.90002441406250, 522.20001220703125, 520.29998779296875, 520.90002441406250, 521.29998779296875, 522.20001220703125, 520.40002441406250, 521.29998779296875, 522.10003662109375, 520.40002441406250, 522.10003662109375, 520.40002441406250, 521.29998779296875, 522.10003662109375, 520.40002441406250, 521.29998779296875, 522.10003662109375, 520.20001220703125, 521.29998779296875, 522.10003662109375, 520.10003662109375, 521.29998779296875, 522.10003662109375, 520.10003662109375, 520.79998779296875, 521.29998779296875, 522.29998779296875, 520.10003662109375, 521.00000000000000, 521.40002441406250, 522.29998779296875, 520.40002441406250, 521.00000000000000, 521.29998779296875, 522.29998779296875, 520.40002441406250, 521.00000000000000, 521.29998779296875, 522.10003662109375, 520.29998779296875, 521.29998779296875, 522.00000000000000, 520.20001220703125, 521.29998779296875, 522.00000000000000, 520.20001220703125, 521.20001220703125, 522.00000000000000, 520.29998779296875, 521.20001220703125, 522.00000000000000, 520.29998779296875, 521.00000000000000, 522.00000000000000, 522.29998779296875, 520.29998779296875, 521.00000000000000, 522.00000000000000, 522.29998779296875, 520.29998779296875, 520.90002441406250, 522.29998779296875, 520.10003662109375, 520.90002441406250, 522.29998779296875, 520.10003662109375, 521.00000000000000, 522.29998779296875, 520.10003662109375, 521.00000000000000, 522.29998779296875, 520.10003662109375, 520.90002441406250, 522.29998779296875, 520.00000000000000, 520.90002441406250, 522.10003662109375, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.20001220703125, 520.40002441406250, 520.90002441406250, 522.10003662109375, 520.40002441406250, 520.40002441406250, 522.29998779296875, 520.40002441406250, 521.00000000000000, 522.29998779296875, 520.40002441406250, 521.00000000000000, 522.29998779296875, 520.00000000000000, 520.40002441406250, 520.90002441406250, 522.29998779296875, 520.00000000000000, 520.29998779296875, 520.90002441406250, 522.20001220703125, 520.10003662109375, 521.10003662109375, 522.20001220703125)
  expected_int2 <- c(774, 974, 510, 746, 1082, 647, 750, 1155, 771, 794, 1131, 849, 930, 1012, 924, 1068, 956, 949, 1022, 1039, 915, 871, 1156, 918, 786, 1170, 1033, 852, 1092, 1114, 925, 1041, 950, 1013, 957, 713, 1067, 811, 644, 1047, 663, 594, 974, 602, 582, 969, 584, 535, 1055, 643, 498, 1104, 737, 563, 1204, 842, 714, 1510, 1083, 842, 2064, 1523, 895, 2463, 1973, 855, 2464, 1971, 2144, 1539, 964, 1871, 1059, 991, 1020, 1598, 865, 1146, 1354, 1239, 1419, 771, 1157, 1605, 804, 950, 1646, 719, 774, 1668, 576, 702, 1420, 622, 554, 719, 1006, 656, 591, 801, 837, 597, 560, 810, 923, 412, 469, 845, 985, 454, 833, 1070, 508, 724, 1224, 572, 614, 1459, 552, 596, 1588, 597, 597, 621, 1455, 673, 535, 569, 1251, 769, 625, 1290, 883, 872, 1461, 1031, 1168, 1506, 1172, 1279, 1405, 1193, 1180, 1378, 1168, 1182, 1438, 968, 1390, 1482, 825, 1526, 1267, 734, 1408, 1047, 1040, 714, 1145, 1298, 1529, 1147, 1464, 589, 1202, 1189, 622, 1239, 684, 867, 665, 1202, 742, 753, 664, 1059, 882, 638, 961)
  
  expected_signal   <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3310., 3390., 2900., 2990.), ncol=2, byrow=TRUE), mz=c(520., 522.4), msLevel=1L, verbose=FALSE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(length(result_extractSignalRawData$messages), 0)
})

test_that('1 matrix rt, 2 data.frame mz (extend rt, verbose)', {
  # Expected signal
  expected_rt1  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz1  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int1 <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  
  expected_rt2  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3330.803, 3332.368, 3332.368, 3332.368, 3333.933, 3333.933, 3333.933, 3335.498, 3335.498, 3335.498, 3337.063, 3337.063, 3337.063, 3337.063, 3338.628, 3338.628, 3338.628, 3340.193, 3340.193, 3340.193, 3341.758, 3341.758, 3341.758, 3343.323, 3343.323, 3343.323, 3344.888, 3344.888, 3344.888, 3346.453, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3354.278, 3355.843, 3355.843, 3355.843, 3357.408, 3357.408, 3357.408, 3358.973, 3358.973, 3358.973, 3360.538, 3360.538, 3360.538, 3362.102, 3362.102, 3362.102, 3363.667, 3363.667, 3363.667, 3365.232, 3365.232, 3365.232, 3366.797, 3366.797, 3366.797, 3368.362, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3379.317, 3379.317, 3380.882, 3380.882, 3380.882, 3380.882, 3382.447, 3382.447, 3382.447, 3384.012, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz2  <- c(460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.30001831054688, 462.20001220703125, 460.30001831054688, 461.30001831054688, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.00000000000000, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.00000000000000, 461.30001831054688, 462.20001220703125, 460.10000610351562, 460.89999389648438, 461.30001831054688, 462.20001220703125, 460.10000610351562, 460.89999389648438, 461.30001831054688, 462.20001220703125, 460.20001220703125, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.10000610351562, 461.20001220703125, 462.20001220703125, 460.00000000000000, 461.10000610351562, 462.30001831054688, 460.00000000000000, 461.00000000000000, 462.30001831054688, 460.00000000000000, 460.30001831054688, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.20001220703125, 461.20001220703125, 462.20001220703125, 460.30001831054688, 461.20001220703125, 462.20001220703125, 460.39999389648438, 461.10000610351562, 462.20001220703125, 460.39999389648438, 461.10000610351562, 462.20001220703125, 460.00000000000000, 460.39999389648438, 461.10000610351562, 462.10000610351562, 460.30001831054688, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.20001220703125, 461.00000000000000, 462.20001220703125, 460.20001220703125, 461.00000000000000, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.10000610351562, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.10000610351562, 462.20001220703125, 460.20001220703125, 461.30001831054688, 462.10000610351562, 460.10000610351562, 461.20001220703125, 462.10000610351562, 460.10000610351562, 461.20001220703125, 462.00000000000000, 460.00000000000000, 461.20001220703125, 462.00000000000000, 462.39999389648438, 460.00000000000000, 461.00000000000000, 462.00000000000000, 462.39999389648438, 460.00000000000000, 461.00000000000000, 462.10000610351562, 462.30001831054688, 460.00000000000000, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.10000610351562, 462.10000610351562, 460.10000610351562, 461.30001831054688, 462.30001831054688, 460.10000610351562, 461.30001831054688, 462.30001831054688)
  expected_int2 <- c(308, 452, 2757, 250, 242, 499, 3260, 311, 547, 3423, 292, 356, 552, 3396, 372, 372, 547, 3410, 427, 536, 3420, 428, 653, 3286, 439, 694, 3117, 467, 582, 2929, 544, 438, 430, 2608, 636, 336, 336, 2204, 722, 1928, 738, 308, 1768, 613, 334, 1561, 483, 330, 1282, 389, 288, 1071, 343, 260, 964, 266, 229, 290, 952, 280, 372, 991, 333, 376, 939, 342, 293, 799, 339, 225, 717, 357, 235, 809, 371, 241, 961, 277, 372, 229, 1084, 393, 276, 1042, 449, 355, 953, 527, 432, 944, 537, 476, 1038, 476, 561, 1206, 418, 649, 1321, 421, 705, 1422, 517, 697, 1551, 646, 646, 1765, 705, 603, 1923, 629, 618, 2032, 592, 643, 1956, 579, 637, 1763, 544, 554, 1578, 445, 427, 1427, 422, 359, 1399, 481, 409, 1307, 543, 494, 1119, 479, 463, 982, 807, 444, 380, 870, 812, 493, 362, 857, 828, 592, 364, 923, 644, 394, 914, 616, 385, 843, 518, 379, 798, 383, 467, 790)
  
  expected_signal   <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  expected_messages <- c('"rt" or "mz" is a matrix/data.frame of 1 row, rows will be duplicated to match the other input\n', 'Reading data from 2 windows\n')
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3310., 3390.), ncol=2, byrow=TRUE), mz=data.frame(matrix(c(520., 522.4, 460., 462.4), ncol=2, byrow=TRUE)), msLevel=1L, verbose=TRUE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages (skip timing)
  expect_equal(length(result_extractSignalRawData$messages), 3)
  expect_equal(result_extractSignalRawData$messages[1:2], expected_messages)
})

test_that('2 data.frame rt, 1 matrix mz (extend mz, verbose)', {
  # Expected signal
  expected_rt1  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz1  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int1 <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  
  expected_rt2  <- c(2900.441, 2900.441, 2900.441, 2902.006, 2902.006, 2902.006, 2903.571, 2903.571, 2903.571, 2905.136, 2905.136, 2905.136, 2906.701, 2906.701, 2906.701, 2908.266, 2908.266, 2908.266, 2909.831, 2909.831, 2909.831, 2911.396, 2911.396, 2911.396, 2912.961, 2912.961, 2912.961, 2914.526, 2914.526, 2914.526, 2916.091, 2916.091, 2916.091, 2917.656, 2917.656, 2917.656, 2919.221, 2919.221, 2919.221, 2920.786, 2920.786, 2920.786, 2922.351, 2922.351, 2922.351, 2923.916, 2923.916, 2923.916, 2925.48, 2925.48, 2925.48, 2927.045, 2927.045, 2927.045, 2928.61, 2928.61, 2928.61, 2930.175, 2930.175, 2930.175, 2931.74, 2931.74, 2931.74, 2933.305, 2933.305, 2933.305, 2934.87, 2934.87, 2936.435, 2936.435, 2936.435, 2938, 2938, 2938, 2938, 2939.565, 2939.565, 2939.565, 2941.13, 2941.13, 2942.695, 2942.695, 2942.695, 2944.26, 2944.26, 2944.26, 2945.825, 2945.825, 2945.825, 2947.39, 2947.39, 2947.39, 2948.955, 2948.955, 2948.955, 2948.955, 2950.52, 2950.52, 2950.52, 2950.52, 2952.085, 2952.085, 2952.085, 2952.085, 2953.65, 2953.65, 2953.65, 2953.65, 2955.215, 2955.215, 2955.215, 2956.78, 2956.78, 2956.78, 2958.344, 2958.344, 2958.344, 2959.909, 2959.909, 2959.909, 2961.474, 2961.474, 2961.474, 2961.474, 2963.039, 2963.039, 2963.039, 2963.039, 2964.604, 2964.604, 2964.604, 2966.169, 2966.169, 2966.169, 2967.734, 2967.734, 2967.734, 2969.299, 2969.299, 2969.299, 2970.864, 2970.864, 2970.864, 2972.429, 2972.429, 2972.429, 2973.994, 2973.994, 2973.994, 2975.559, 2975.559, 2975.559, 2977.124, 2977.124, 2977.124, 2978.689, 2978.689, 2978.689, 2978.689, 2980.254, 2981.819, 2981.819, 2983.384, 2983.384, 2983.384, 2984.949, 2984.949, 2984.949, 2986.514, 2986.514, 2986.514, 2986.514, 2988.079, 2988.079, 2988.079, 2988.079, 2989.644, 2989.644, 2989.644)
  expected_mz2  <- c(520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.10003662109375, 522.10003662109375, 520.20001220703125, 521.10003662109375, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.20001220703125, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.10003662109375, 520.10003662109375, 521.10003662109375, 522.20001220703125, 520.20001220703125, 521.10003662109375, 522.20001220703125, 520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.00000000000000, 522.10003662109375, 520.20001220703125, 521.00000000000000, 522.00000000000000, 520.20001220703125, 521.10003662109375, 522.00000000000000, 520.10003662109375, 521.20001220703125, 522.20001220703125, 520.10003662109375, 521.10003662109375, 522.20001220703125, 520.10003662109375, 521.10003662109375, 522.00000000000000, 520.10003662109375, 521.10003662109375, 522.00000000000000, 520.10003662109375, 521.10003662109375, 521.90002441406250, 520.10003662109375, 521.00000000000000, 521.90002441406250, 520.20001220703125, 520.90002441406250, 521.90002441406250, 520.20001220703125, 520.90002441406250, 521.90002441406250, 520.20001220703125, 520.90002441406250, 520.20001220703125, 520.90002441406250, 522.20001220703125, 520.29998779296875, 520.90002441406250, 521.29998779296875, 522.20001220703125, 520.40002441406250, 521.29998779296875, 522.10003662109375, 520.40002441406250, 522.10003662109375, 520.40002441406250, 521.29998779296875, 522.10003662109375, 520.40002441406250, 521.29998779296875, 522.10003662109375, 520.20001220703125, 521.29998779296875, 522.10003662109375, 520.10003662109375, 521.29998779296875, 522.10003662109375, 520.10003662109375, 520.79998779296875, 521.29998779296875, 522.29998779296875, 520.10003662109375, 521.00000000000000, 521.40002441406250, 522.29998779296875, 520.40002441406250, 521.00000000000000, 521.29998779296875, 522.29998779296875, 520.40002441406250, 521.00000000000000, 521.29998779296875, 522.10003662109375, 520.29998779296875, 521.29998779296875, 522.00000000000000, 520.20001220703125, 521.29998779296875, 522.00000000000000, 520.20001220703125, 521.20001220703125, 522.00000000000000, 520.29998779296875, 521.20001220703125, 522.00000000000000, 520.29998779296875, 521.00000000000000, 522.00000000000000, 522.29998779296875, 520.29998779296875, 521.00000000000000, 522.00000000000000, 522.29998779296875, 520.29998779296875, 520.90002441406250, 522.29998779296875, 520.10003662109375, 520.90002441406250, 522.29998779296875, 520.10003662109375, 521.00000000000000, 522.29998779296875, 520.10003662109375, 521.00000000000000, 522.29998779296875, 520.10003662109375, 520.90002441406250, 522.29998779296875, 520.00000000000000, 520.90002441406250, 522.10003662109375, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.10003662109375, 521.00000000000000, 522.10003662109375, 520.20001220703125, 520.40002441406250, 520.90002441406250, 522.10003662109375, 520.40002441406250, 520.40002441406250, 522.29998779296875, 520.40002441406250, 521.00000000000000, 522.29998779296875, 520.40002441406250, 521.00000000000000, 522.29998779296875, 520.00000000000000, 520.40002441406250, 520.90002441406250, 522.29998779296875, 520.00000000000000, 520.29998779296875, 520.90002441406250, 522.20001220703125, 520.10003662109375, 521.10003662109375, 522.20001220703125)
  expected_int2 <- c(774, 974, 510, 746, 1082, 647, 750, 1155, 771, 794, 1131, 849, 930, 1012, 924, 1068, 956, 949, 1022, 1039, 915, 871, 1156, 918, 786, 1170, 1033, 852, 1092, 1114, 925, 1041, 950, 1013, 957, 713, 1067, 811, 644, 1047, 663, 594, 974, 602, 582, 969, 584, 535, 1055, 643, 498, 1104, 737, 563, 1204, 842, 714, 1510, 1083, 842, 2064, 1523, 895, 2463, 1973, 855, 2464, 1971, 2144, 1539, 964, 1871, 1059, 991, 1020, 1598, 865, 1146, 1354, 1239, 1419, 771, 1157, 1605, 804, 950, 1646, 719, 774, 1668, 576, 702, 1420, 622, 554, 719, 1006, 656, 591, 801, 837, 597, 560, 810, 923, 412, 469, 845, 985, 454, 833, 1070, 508, 724, 1224, 572, 614, 1459, 552, 596, 1588, 597, 597, 621, 1455, 673, 535, 569, 1251, 769, 625, 1290, 883, 872, 1461, 1031, 1168, 1506, 1172, 1279, 1405, 1193, 1180, 1378, 1168, 1182, 1438, 968, 1390, 1482, 825, 1526, 1267, 734, 1408, 1047, 1040, 714, 1145, 1298, 1529, 1147, 1464, 589, 1202, 1189, 622, 1239, 684, 867, 665, 1202, 742, 753, 664, 1059, 882, 638, 961)
  
  expected_signal   <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  expected_messages <- c('"rt" or "mz" is a matrix/data.frame of 1 row, rows will be duplicated to match the other input\n', 'Reading data from 2 windows\n')
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=data.frame(matrix(c(3310., 3390., 2900., 2990.), ncol=2, byrow=TRUE)), mz=matrix(c(520., 522.4), ncol=2, byrow=TRUE), msLevel=1L, verbose=TRUE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages (skip timing)
  expect_equal(length(result_extractSignalRawData$messages), 3)
  expect_equal(result_extractSignalRawData$messages[1:2], expected_messages)
})

test_that('no rt window defined, 2 data.frame mz (all rt range + extend rt)', {
  # Expected signal
  expected_rt1  <- c(2501.378, 2501.378, 2502.943, 2504.508, 2506.073, 2507.638, 2509.203, 2509.203, 2510.768, 2510.768, 2512.333, 2512.333, 2513.898, 2513.898, 2515.463, 2517.028, 2518.593, 2520.158, 2520.158, 2521.722, 2521.722, 2523.287, 2523.287, 2524.852, 2524.852, 2526.417, 2526.417, 2527.982, 2527.982, 2529.547, 2529.547, 2531.112, 2531.112, 2532.677, 2532.677, 2534.242, 2535.807, 2537.372, 2538.937, 2540.502, 2542.067, 2543.632, 2545.197, 2546.762, 2548.327, 2549.892, 2551.457, 2551.457, 2553.022, 2553.022, 2554.586, 2556.151, 2557.716, 2559.281, 2559.281, 2560.846, 2560.846, 2562.411, 2562.411, 2563.976, 2565.541, 2567.106, 2568.671, 2570.236, 2570.236, 2571.801, 2571.801, 2573.366, 2573.366, 2574.931, 2574.931, 2576.496, 2578.061, 2579.626, 2581.191, 2582.756, 2584.321, 2585.886, 2587.451, 2589.015, 2590.58, 2592.145, 2593.71, 2593.71, 2595.275, 2595.275, 2596.84, 2598.405, 2599.97, 2601.535, 2601.535, 2603.1, 2603.1, 2604.665, 2604.665, 2606.23, 2607.795, 2609.36, 2610.925, 2612.49, 2612.49, 2614.055, 2615.62, 2617.185, 2618.75, 2620.315, 2621.879, 2623.444, 2625.009, 2626.574, 2628.139, 2629.704, 2631.269, 2632.834, 2634.399, 2635.964, 2637.529, 2637.529, 2639.094, 2639.094, 2640.659, 2642.224, 2642.224, 2643.789, 2643.789, 2645.354, 2645.354, 2646.919, 2648.484, 2650.049, 2651.614, 2653.179, 2654.744, 2656.308, 2656.308, 2657.873, 2657.873, 2659.438, 2659.438, 2661.003, 2661.003, 2662.568, 2662.568, 2662.568, 2664.133, 2664.133, 2664.133, 2665.698, 2665.698, 2667.263, 2667.263, 2668.828, 2670.393, 2671.958, 2673.523, 2675.088, 2675.088, 2676.653, 2678.218, 2679.783, 2681.348, 2681.348, 2682.913, 2684.478, 2686.043, 2687.608, 2689.172, 2690.737, 2690.737, 2692.302, 2692.302, 2693.867, 2693.867, 2695.432, 2695.432, 2696.997, 2696.997, 2698.562, 2700.127, 2700.127, 2701.692, 2701.692, 2703.257, 2704.822, 2706.387, 2707.952, 2709.517, 2711.082, 2712.647, 2712.647, 2714.212, 2714.212, 2715.777, 2715.777, 2717.342, 2717.342, 2718.907, 2720.472, 2722.037, 2723.601, 2725.166, 2726.731, 2728.296, 2729.861, 2731.426, 2732.991, 2732.991, 2734.556, 2736.121, 2737.686, 2739.251, 2740.816, 2740.816, 2742.381, 2742.381, 2743.946, 2745.511, 2747.076, 2748.641, 2750.206, 2751.771, 2753.336, 2754.901, 2756.465, 2756.465, 2758.03, 2758.03, 2759.595, 2759.595, 2761.16, 2761.16, 2762.725, 2762.725, 2764.29, 2764.29, 2765.855, 2767.42, 2767.42, 2768.985, 2768.985, 2770.55, 2770.55, 2772.115, 2772.115, 2773.68, 2775.245, 2776.81, 2778.375, 2779.94, 2779.94, 2781.505, 2781.505, 2783.07, 2783.07, 2784.635, 2787.765, 2789.33, 2789.33, 2790.894, 2790.894, 2790.894, 2792.459, 2792.459, 2792.459, 2794.024, 2794.024, 2795.589, 2795.589, 2797.154, 2797.154, 2798.719, 2798.719, 2800.284, 2800.284, 2801.849, 2801.849, 2803.414, 2803.414, 2804.979, 2804.979, 2806.544, 2806.544, 2806.544, 2808.109, 2809.674, 2811.239, 2812.804, 2814.369, 2815.934, 2817.499, 2819.064, 2820.629, 2822.194, 2822.194, 2823.758, 2825.323, 2826.888, 2828.453, 2830.018, 2830.018, 2831.583, 2833.148, 2833.148, 2834.713, 2834.713, 2836.278, 2836.278, 2837.843, 2837.843, 2839.408, 2840.973, 2840.973, 2842.538, 2842.538, 2844.103, 2845.668, 2847.233, 2847.233, 2848.798, 2848.798, 2850.363, 2850.363, 2851.928, 2851.928, 2853.493, 2853.493, 2855.058, 2855.058, 2856.623, 2858.187, 2859.752, 2861.317, 2862.882, 2864.447, 2866.012, 2866.012, 2867.577, 2867.577, 2869.142, 2869.142, 2870.707, 2870.707, 2872.272, 2872.272, 2873.837, 2875.402, 2876.967, 2878.532, 2878.532, 2880.097, 2880.097, 2881.662, 2881.662, 2883.227, 2883.227, 2884.792, 2886.357, 2886.357, 2887.922, 2887.922, 2889.487, 2889.487, 2891.051, 2891.051, 2892.616, 2892.616, 2894.181, 2895.746, 2897.311, 2898.876, 2900.441, 2900.441, 2902.006, 2902.006, 2903.571, 2905.136, 2906.701, 2908.266, 2908.266, 2909.831, 2911.396, 2912.961, 2914.526, 2916.091, 2916.091, 2917.656, 2917.656, 2919.221, 2919.221, 2920.786, 2922.351, 2923.916, 2925.48, 2927.045, 2928.61, 2930.175, 2930.175, 2931.74, 2931.74, 2933.305, 2933.305, 2934.87, 
                    2934.87, 2936.435, 2936.435, 2938, 2938, 2939.565, 2941.13, 2942.695, 2944.26, 2945.825, 2947.39, 2948.955, 2948.955, 2950.52, 2950.52, 2952.085, 2952.085, 2953.65, 2953.65, 2955.215, 2956.78, 2958.344, 2959.909, 2961.474, 2961.474, 2963.039, 2963.039, 2964.604, 2964.604, 2966.169, 2966.169, 2967.734, 2967.734, 2969.299, 2969.299, 2970.864, 2970.864, 2972.429, 2972.429, 2973.994, 2973.994, 2975.559, 2975.559, 2977.124, 2977.124, 2978.689, 2978.689, 2978.689, 2980.254, 2981.819, 2983.384, 2983.384, 2984.949, 2984.949, 2986.514, 2986.514, 2986.514, 2988.079, 2988.079, 2988.079, 2989.644, 2991.209, 2992.773, 2994.338, 2995.903, 2995.903, 2997.468, 2997.468, 2999.033, 2999.033, 3000.598, 3000.598, 3002.163, 3003.728, 3005.293, 3006.858, 3008.423, 3009.988, 3009.988, 3011.553, 3011.553, 3013.118, 3013.118, 3014.683, 3014.683, 3016.248, 3017.813, 3019.378, 3020.943, 3020.943, 3022.508, 3022.508, 3024.073, 3024.073, 3025.637, 3025.637, 3027.202, 3028.767, 3030.332, 3031.897, 3031.897, 3033.462, 3033.462, 3033.462, 3035.027, 3035.027, 3036.592, 3038.157, 3039.722, 3041.287, 3042.852, 3044.417, 3045.982, 3047.547, 3047.547, 3049.112, 3049.112, 3049.112, 3050.677, 3050.677, 3052.242, 3053.807, 3055.372, 3056.937, 3058.502, 3058.502, 3060.066, 3060.066, 3061.631, 3061.631, 3063.196, 3064.761, 3066.326, 3067.891, 3069.456, 3071.021, 3072.586, 3074.151, 3075.716, 3077.281, 3078.846, 3080.411, 3081.976, 3083.541, 3085.106, 3086.671, 3088.236, 3089.801, 3091.366, 3092.93, 3092.93, 3094.495, 3094.495, 3096.06, 3097.625, 3099.19, 3099.19, 3100.755, 3100.755, 3102.32, 3103.885, 3105.45, 3107.015, 3107.015, 3108.58, 3108.58, 3110.145, 3111.71, 3113.275, 3114.84, 3116.405, 3117.97, 3119.535, 3121.1, 3122.665, 3124.23, 3125.794, 3127.359, 3128.924, 3130.489, 3132.054, 3133.619, 3135.184, 3136.749, 3138.314, 3139.879, 3141.444, 3143.009, 3144.574, 3146.139, 3147.704, 3149.269, 3150.834, 3152.399, 3153.964, 3155.529, 3157.094, 3158.659, 3160.223, 3161.788, 3163.353, 3164.918, 3166.483, 3168.048, 3169.613, 3171.178, 3172.743, 3174.308, 3175.873, 3177.438, 3179.003, 3180.568, 3182.133, 3183.698, 3185.263, 3186.828, 3188.393, 3189.958, 3191.523, 3193.087, 3194.652, 3196.217, 3197.782, 3199.347, 3200.912, 3202.477, 3204.042, 3205.607, 3207.172, 3208.737, 3210.302, 3211.867, 3213.432, 3214.997, 3216.562, 3218.127, 3219.692, 3221.257, 3222.822, 3224.387, 3225.952, 3227.516, 3229.081, 3230.646, 3230.646, 3232.211, 3233.776, 3235.341, 3236.906, 3238.471, 3240.036, 3241.601, 3243.166, 3244.731, 3246.296, 3247.861, 3249.426, 3250.991, 3252.556, 3254.121, 3255.686, 3257.251, 3258.816, 3260.38, 3261.945, 3263.51, 3265.075, 3266.64, 3268.205, 3269.77, 3271.335, 3272.9, 3274.465, 3276.03, 3277.595, 3279.16, 3280.725, 3282.29, 3283.855, 3285.42, 3286.985, 3286.985, 3288.55, 3288.55, 3290.115, 3290.115, 3291.68, 3293.245, 3294.809, 3296.374, 3297.939, 3299.504, 3301.069, 3302.634, 3304.199, 3305.764, 3307.329, 3308.894, 3310.459, 3312.024, 3312.024, 3313.589, 3313.589, 3315.154, 3316.719, 3318.284, 3319.849, 3321.414, 3322.979, 3324.544, 3326.109, 3327.673, 3329.238, 3329.238, 3330.803, 3332.368, 3333.933, 3335.498, 3337.063, 3338.628, 3340.193, 3341.758, 3343.323, 3344.888, 3346.453, 3348.018, 3348.018, 3349.583, 3349.583, 3351.148, 3351.148, 3352.713, 3352.713, 3354.278, 3355.843, 3357.408, 3358.973, 3360.538, 3362.102, 3363.667, 3365.232, 3366.797, 3368.362, 3369.927, 3369.927, 3371.492, 3371.492, 3373.057, 3374.622, 3376.187, 3377.752, 3379.317, 3380.882, 3382.447, 3384.012, 3385.577, 3387.142, 3388.707, 3390.272, 3391.837, 3393.402, 3393.402, 3394.966, 3394.966, 3396.531, 3398.096, 3399.661, 3401.226, 3401.226, 3402.791, 3402.791, 3404.356, 3404.356, 3404.356, 3405.921, 3407.486, 3409.051, 3410.616, 3412.181, 3413.746, 3415.311, 3415.311, 3416.876, 3416.876, 3418.441, 
                    3420.006, 3421.571, 3423.136, 3424.701, 3426.266, 3427.831, 3429.395, 3430.96, 3432.525, 3434.09, 3435.655, 3437.22, 3438.785, 3440.35, 3441.915, 3443.48, 3445.045, 3446.61, 3448.175, 3449.74, 3451.305, 3451.305, 3452.87, 3452.87, 3454.435, 3454.435, 3456, 3456, 3457.565, 3457.565, 3459.13, 3460.695, 3462.259, 3463.824, 3465.389, 3466.954, 3468.519, 3470.084, 3471.649, 3473.214, 3474.779, 3476.344, 3477.909, 3479.474, 3481.039, 3482.604, 3484.169, 3485.734, 3487.299, 3488.864, 3490.429, 3491.994, 3493.559, 3495.124, 3496.688, 3498.253, 3499.818, 3501.383, 3502.948, 3502.948, 3504.513, 3506.078, 3507.643, 3509.208, 3510.773, 3512.338, 3513.903, 3515.468, 3517.033, 3518.598, 3520.163, 3521.728, 3523.293, 3524.858, 3526.423, 3527.988, 3529.552, 3531.117, 3532.682, 3534.247, 3535.812, 3537.377, 3538.942, 3540.507, 3542.072, 3543.637, 3545.202, 3546.767, 3548.332, 3549.897, 3551.462, 3553.027, 3554.592, 3556.157, 3557.722, 3559.287, 3560.852, 3562.417, 3563.981, 3565.546, 3567.111, 3568.676, 3570.241, 3571.806, 3573.371, 3574.936, 3576.501, 3578.066, 3579.631, 3581.196, 3582.761, 3584.326, 3585.891, 3587.456, 3589.021, 3590.586, 3592.151, 3593.716, 3595.281, 3596.845, 3598.41, 3599.975, 3601.54, 3603.105, 3603.105, 3604.67, 3606.235, 3607.8, 3609.365, 3610.93, 3612.495, 3614.06, 3615.625, 3617.19, 3618.755, 3620.32, 3621.885, 3623.45, 3625.015, 3626.58, 3628.145, 3629.71, 3631.274, 3632.839, 3634.404, 3635.969, 3637.534, 3639.099, 3640.664, 3642.229, 3643.794, 3645.359, 3646.924, 3648.489, 3650.054, 3651.619, 3653.184, 3654.749, 3656.314, 3657.879, 3659.444, 3661.009, 3662.574, 3664.138, 3665.703, 3667.268, 3668.833, 3670.398, 3671.963, 3673.528, 3675.093, 3676.658, 3678.223, 3679.788, 3681.353, 3682.918, 3684.483, 3686.048, 3687.613, 3689.178, 3690.743, 3692.308, 3693.873, 3695.438, 3697.003, 3698.567, 3700.132, 3701.697, 3703.262, 3704.827, 3706.392, 3707.957, 3709.522, 3711.087, 3712.652, 3714.217, 3715.782, 3717.347, 3718.912, 3720.477, 3722.042, 3723.607, 3725.172, 3726.737, 3728.302, 3729.867, 3731.431, 3732.996, 3734.561, 3736.126, 3737.691, 3739.256, 3740.821, 3742.386, 3743.951, 3745.516, 3747.081, 3748.646, 3750.211, 3751.776, 3753.341, 3754.906, 3756.471, 3758.036, 3759.601, 3761.166, 3762.731, 3764.296, 3765.86, 3767.425, 3768.99, 3770.555, 3772.12, 3773.685, 3775.25, 3776.815, 3778.38, 3779.945, 3781.51, 3783.075, 3784.64, 3786.205, 3787.77, 3789.335, 3790.9, 3792.465, 3794.03, 3795.595, 3797.16, 3798.724, 3800.289, 3801.854, 3803.419, 3804.984, 3806.549, 3808.114, 3809.679, 3811.244, 3812.809, 3814.374, 3815.939, 3817.504, 3819.069, 3820.634, 3822.199, 3823.764, 3825.329, 3826.894, 3828.459, 3830.024, 3831.589, 3833.153, 3834.718, 3836.283, 3837.848, 3839.413, 3840.978, 3842.543, 3844.108, 3845.673, 3847.238, 3848.803, 3850.368, 3851.933, 3853.498, 3855.063, 3856.628, 3858.193, 3859.758, 3861.323, 3862.888, 3864.453, 3866.017, 3867.582, 3867.582, 3869.147, 3869.147, 3870.712, 3872.277, 3873.842, 3875.407, 3876.972, 3878.537, 3880.102, 3881.667, 3883.232, 3884.797, 3886.362, 3887.927, 3889.492, 3891.057, 3892.622, 3894.187, 3895.752, 3897.317, 3898.882, 3900.446, 3902.011, 3903.576, 3905.141, 3906.706, 3908.271, 3909.836, 3911.401, 3912.966, 3914.531, 3916.096, 3917.661, 3919.226, 3920.791, 3922.356, 3923.921, 3925.486, 3927.051, 3928.616, 3930.181, 3931.746, 3933.31, 3934.875, 3936.44, 3938.005, 3939.57, 3941.135, 3942.7, 3944.265, 3945.83, 3947.395, 3948.96, 3950.525, 3952.09, 3953.655, 3955.22, 3956.785, 3958.35, 3959.915, 3961.48, 3963.045, 3964.61, 3966.175, 3967.739, 3969.304, 3970.869, 3972.434, 3973.999, 3975.564, 3977.129, 3978.694, 3980.259, 3981.824, 3983.389, 3984.954, 3986.519, 3988.084, 3989.649, 3991.214, 3992.779, 3994.344, 3995.909, 3997.474, 3999.039, 4000.603, 4002.168, 4003.733, 4005.298, 4006.863, 4008.428, 4009.993, 
                    4011.558, 4013.123, 4014.688, 4016.253, 4017.818, 4019.383, 4020.948, 4022.513, 4024.078, 4025.643, 4027.208, 4028.773, 4030.338, 4031.903, 4033.467, 4035.032, 4036.597, 4038.162, 4039.727, 4041.292, 4042.857, 4044.422, 4045.987, 4047.552, 4049.117, 4050.682, 4052.247, 4053.812, 4055.377, 4056.942, 4058.507, 4060.072, 4061.637, 4063.202, 4064.767, 4066.332, 4067.896, 4069.461, 4071.026, 4072.591, 4074.156, 4075.721, 4077.286, 4078.851, 4080.416, 4081.981, 4083.546, 4085.111, 4086.676, 4088.241, 4089.806, 4091.371, 4092.936, 4094.501, 4096.066, 4097.631, 4099.196, 4100.76, 4102.325, 4103.89, 4105.455, 4107.02, 4108.585, 4110.15, 4111.715, 4113.28, 4114.845, 4116.41, 4117.975, 4119.54, 4121.105, 4122.67, 4124.235, 4125.8, 4127.365, 4128.93, 4130.495, 4132.06, 4133.625, 4135.189, 4136.754, 4138.319, 4139.884, 4141.449, 4143.014, 4143.014, 4144.579, 4146.144, 4147.709, 4149.274, 4155.534, 4157.099, 4158.664, 4160.229, 4163.359, 4164.924, 4166.489, 4168.053, 4169.618, 4169.618, 4171.183, 4171.183, 4172.748, 4172.748, 4174.313, 4175.878, 4405.926, 4407.491, 4409.056, 4418.446, 4473.219)
  expected_mz1  <- c(520, 521, 520, 520, 520, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2000122, 521, 520.1000366, 520.2999878, 520.4000244, 520.4000244, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2999878, 521, 520.2999878, 521, 520.2999878, 520.9000244, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2000122, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2999878, 520.2000122, 521, 520.1000366, 521, 520.1000366, 520.2000122, 520.1000366, 520.1000366, 520.7999878, 520, 520.7999878, 520, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.1000366, 521, 520, 521, 520, 521, 520, 521, 520, 520, 520, 520, 520, 520, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.9000244, 520.1000366, 520.9000244, 520, 520, 520, 520, 520.7999878, 520.1000366, 520.7999878, 520.1000366, 520.9000244, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520, 520.1000366, 520.2000122, 520.2999878, 520.4000244, 520.4000244, 520.4000244, 521, 520.2999878, 521, 520.1000366, 520.1000366, 521, 520.1000366, 521, 520.1000366, 521, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 521, 520.1000366, 521, 520.1000366, 521, 520.1000366, 520.9000244, 520, 520.4000244, 520.9000244, 520.1000366, 520.5, 521, 520.1000366, 521, 520.1000366, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 521, 520.1000366, 520.1000366, 520, 520, 521, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 521, 520.1000366, 521, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.1000366, 521, 520.1000366, 520.1000366, 521, 520.1000366, 521, 520, 520, 520, 520.1000366, 520.1000366, 520.1000366, 520, 521, 520.1000366, 521, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520, 520, 520, 520, 520.1000366, 520.2999878, 520.1000366, 520, 520, 520, 520, 520.4000244, 520, 520.4000244, 520, 520, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2999878, 521, 520.2999878, 521, 520.2000122, 521, 520.2000122, 520.9000244, 520.1000366, 520.9000244, 520, 520.9000244, 520, 520, 520.2999878, 520, 520.4000244, 520, 520.4000244, 520, 520.2999878, 520.4000244, 520.4000244, 520.2000122, 520.1000366, 520.1000366, 521, 520.1000366, 520.9000244, 520, 520.9000244, 520, 520.4000244, 520, 520.4000244, 520, 520.4000244, 521, 520, 520.4000244, 520.9000244, 520.4000244, 520.9000244, 520.4000244, 520.9000244, 520.1000366, 520.7999878, 520.1000366, 520.7999878, 520.1000366, 520.7999878, 520, 520.7999878, 520, 520.7999878, 520, 520.7999878, 520, 520.2999878, 520.7999878, 520.2999878, 520.2999878, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520, 520, 520.1000366, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 520.2999878, 521, 520.2999878, 521, 520.1000366, 521, 520.1000366, 521, 520.1000366, 520.1000366, 520.9000244, 520, 520.7999878, 520.1000366, 520.2000122, 520.2000122, 520.9000244, 520.2999878, 520.9000244, 520.2999878, 520.9000244, 520.2999878, 520.9000244, 520.4000244, 520.9000244, 520.2000122, 521, 520.1000366, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.9000244, 520.2999878, 520.9000244, 520.2999878, 520.9000244, 520.4000244, 521, 520.2000122, 521, 520.2000122, 520.2000122, 520.2000122, 520, 521, 520, 521, 520, 520.5, 520, 520.4000244, 520, 520, 521, 520, 521, 520.1000366, 521, 520.2000122, 521, 520.1000366, 521, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.2000122,
                    520.1000366, 520.1000366, 521, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 521, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2999878, 520.9000244, 520.4000244, 520.4000244, 520.4000244, 520.4000244, 520.2000122, 520.1000366, 520.1000366, 520.7999878, 520.1000366, 521, 520.4000244, 521, 520.4000244, 521, 520.2999878, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 521, 520.2999878, 521, 520.2999878, 520.9000244, 520.1000366, 520.9000244, 520.1000366, 521, 520.1000366, 521, 520.1000366, 520.9000244, 520, 520.9000244, 520.1000366, 521, 520.1000366, 521, 520.1000366, 521, 520.2000122, 520.4000244, 520.9000244, 520.4000244, 520.4000244, 520.4000244, 521, 520.4000244, 521, 520, 520.4000244, 520.9000244, 520, 520.2999878, 520.9000244, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 521, 520.1000366, 521, 520.2000122, 521, 520, 521, 520, 520.1000366, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 521, 520.2000122, 521, 520.2000122, 521, 520.4000244, 520.9000244, 520.4000244, 520.4000244, 520.2999878, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2000122, 520.9000244, 520.2999878, 521, 520.2999878, 520.4000244, 520.4000244, 520.4000244, 521, 520, 520.4000244, 521, 520, 520.2999878, 520, 520, 521, 520.2999878, 520.4000244, 520, 520, 520, 521, 520, 520.2999878, 520.9000244, 520.4000244, 520.7999878, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.1000366, 521, 520.1000366, 521, 520.1000366, 521, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.1000366, 520.1000366, 520.1000366, 521, 520.1000366, 521, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 521, 520.1000366, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520, 520.4000244, 520, 520, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366,
                    520.2000122, 520.2000122, 520.1000366, 520.1000366, 521, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.4000244, 520, 520.2999878, 520, 520.2999878, 520, 520.2999878, 520, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.9000244, 520, 521, 520.1000366, 520.1000366, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.9000244, 520.2000122, 520.9000244, 520.1000366, 520.2000122, 520.2000122, 520.2999878, 521, 520.2999878, 521, 520.1000366, 520.2999878, 521, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2999878, 520.9000244, 520.2999878, 520.9000244, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.1000366, 520, 520, 520, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.9000244, 520.2000122, 521, 520.2000122, 521, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520, 520.1000366, 520.1000366, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.1000366, 520.1000366, 520, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.1000366, 520.1000366, 520, 520, 520, 520.1000366, 520.1000366, 520.1000366, 520.2999878, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122,
                    520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 521, 520.2000122, 521, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.4000244, 520.4000244, 520.2999878, 520.2999878, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520, 520, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2999878, 520.2999878, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.1000366, 520.1000366, 520.2000122, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.2000122, 520.1000366, 520.1000366, 520.2000122, 520.2999878, 520, 520.1000366, 520.2000122, 520.2999878, 520.4000244, 520.4000244, 520.4000244, 520.4000244, 520.2999878, 520.4000244, 520.4000244, 520.4000244, 520.2999878, 520.2999878, 520.2999878, 520.2999878, 520.4000244, 520.5, 520.5, 520, 520.5, 520.2000122, 520.2000122, 520.2000122, 520.1000366, 520.4000244, 520.4000244, 520.4000244, 520.4000244, 520.2999878, 520.2000122, 520.2000122, 520, 520.5, 521, 520.4000244, 521, 520.2000122, 521, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.2000122, 520.7000122, 520.7999878)
  expected_int1 <- c(645, 476, 718, 708, 567, 516, 482, 410, 477, 565, 486, 595, 475, 478, 528, 499, 510, 585, 676, 630, 819, 750, 817, 826, 688, 854, 545, 836, 522, 819, 626, 748, 709, 720, 683, 668, 653, 704, 807, 791, 726, 792, 846, 717, 610, 677, 745, 536, 873, 520, 1033, 1050, 922, 723, 353, 637, 349, 514, 516, 621, 609, 534, 488, 503, 542, 615, 543, 748, 489, 848, 388, 892, 898, 879, 909, 974, 948, 872, 808, 848, 852, 873, 919, 407, 917, 387, 892, 824, 660, 565, 404, 600, 484, 696, 459, 745, 743, 709, 704, 740, 666, 775, 822, 877, 880, 854, 799, 828, 887, 904, 853, 790, 783, 771, 720, 796, 779, 853, 634, 794, 589, 602, 457, 593, 426, 603, 399, 618, 587, 568, 635, 743, 782, 783, 640, 773, 595, 663, 477, 553, 430, 554, 449, 460, 666, 520, 499, 803, 547, 854, 572, 839, 797, 853, 1071, 1291, 543, 1357, 1259, 1143, 1029, 700, 970, 1013, 1090, 1058, 900, 727, 619, 638, 689, 614, 713, 574, 724, 632, 631, 688, 607, 583, 489, 615, 475, 549, 527, 481, 477, 452, 426, 600, 479, 629, 632, 697, 782, 763, 790, 736, 745, 805, 742, 693, 792, 887, 807, 684, 667, 562, 517, 568, 605, 600, 400, 603, 462, 644, 762, 942, 1084, 1054, 891, 670, 571, 587, 577, 667, 611, 755, 567, 750, 470, 665, 422, 643, 403, 630, 583, 561, 537, 553, 521, 536, 517, 600, 654, 589, 469, 469, 487, 510, 499, 436, 520, 364, 544, 416, 657, 569, 636, 675, 469, 549, 656, 462, 556, 435, 472, 401, 502, 450, 632, 447, 707, 441, 755, 463, 791, 583, 730, 667, 551, 571, 675, 606, 529, 426, 509, 703, 833, 815, 678, 586, 530, 529, 637, 732, 768, 754, 694, 606, 667, 751, 506, 872, 541, 881, 626, 906, 675, 784, 618, 654, 528, 642, 511, 547, 646, 891, 741, 942, 751, 906, 653, 843, 562, 747, 517, 596, 670, 818, 835, 812, 787, 738, 593, 347, 477, 383, 459, 410, 539, 452, 570, 507, 656, 764, 847, 914, 380, 860, 343, 720, 681, 632, 650, 671, 747, 574, 743, 517, 670, 542, 589, 559, 579, 510, 718, 866, 910, 831, 774, 974, 746, 1082, 750, 794, 930, 1068, 956, 1022, 871, 786, 852, 925, 1041, 1013, 957, 1067, 811, 1047, 974, 969, 1055, 1104, 1204, 1510, 1083, 2064, 1523, 2463, 1973, 2464, 1971, 2144, 1539, 1871, 1059, 1598, 1354, 1419, 1605, 1646, 1668, 1420, 622, 1006, 656, 837, 597, 923, 412, 985, 1070, 1224, 1459, 1588, 597, 1455, 673, 1251, 769, 1290, 883, 1461, 1031, 1506, 1172, 1405, 1193, 1378, 1168, 1438, 968, 1482, 825, 1267, 734, 1047, 1040, 714, 1298, 1529, 1464, 589, 1189, 622, 684, 867, 665, 742, 753, 664, 882, 1029, 1014, 861, 702, 510, 610, 614, 561, 788, 612, 822, 645, 659, 713, 785, 819, 786, 575, 729, 641, 657, 623, 757, 626, 970, 1018, 946, 990, 951, 949, 1202, 839, 1219, 812, 1001, 928, 1075, 1076, 945, 890, 686, 761, 899, 677, 658, 738, 836, 670, 552, 544, 607, 716, 755, 575, 701, 665, 679, 799, 698, 911, 1044, 1140, 1252, 1285, 674, 1283, 800, 1312, 900, 1493, 1755, 1911, 2005, 2127, 2459, 2910, 3442, 3730, 3809, 3711, 3591, 3453, 3233, 2933, 2654, 2454, 2232, 2081, 1979, 1228, 1865, 1178, 1712, 1549, 1487, 1197, 1569, 1102, 1692, 1730, 1643, 1500, 1121, 1399, 1309, 1380, 1485, 1709, 1971, 2076, 1911, 1690, 1573, 1556, 1540, 1543, 1494, 1391, 1349, 1349, 1263, 1131, 1120, 1394, 1640, 1719, 1562, 1315, 1046, 950, 1020, 1176, 1161, 976, 688, 519, 257, 579, 545, 621, 897, 1450, 2352, 3750, 5615, 8515, 13127, 19096, 25080, 30416, 35104, 39568, 43128, 45424, 45992, 45512, 44072, 41928, 39192, 36280, 33136, 29672, 26344, 23352, 20448, 17424, 14733, 12618, 10783, 9180, 7880, 6817, 5941, 5474, 5242, 4836, 4216, 3682, 3531, 3821, 4596, 5719, 6879, 2616, 8354, 10547, 13732, 17016, 20272, 24064, 28560, 32184, 33928, 33992, 32840, 31272, 30376, 30456, 29752, 27280, 23704, 20424,
                    18088, 16304, 14397, 12791, 11621, 10800, 9719, 8272, 6871, 5996, 5623, 5374, 5097, 4731, 4311, 3690, 3220, 3131, 1312, 3390, 1363, 3705, 1223, 3783, 3442, 3045, 2537, 2094, 1847, 1760, 1666, 1431, 1255, 1335, 1624, 1850, 1864, 408, 1637, 493, 1466, 1452, 1655, 1828, 2034, 2225, 2485, 2526, 2407, 2186, 925, 2033, 1975, 1997, 1853, 1629, 1420, 1311, 1322, 1360, 1261, 1061, 936, 902, 958, 920, 1178, 997, 1366, 1129, 1416, 1541, 1621, 1499, 1300, 1619, 2171, 2632, 3164, 3686, 3914, 1755, 3679, 1560, 3255, 3230, 3475, 3573, 3427, 3360, 3504, 3751, 3761, 3562, 3472, 3541, 3695, 3396, 1099, 2945, 1170, 2783, 2983, 3189, 3161, 2071, 2725, 2245, 2373, 2379, 2032, 2455, 2642, 2546, 2439, 2484, 2510, 2370, 1630, 2399, 1443, 2503, 2564, 2621, 2670, 2657, 2391, 2025, 1735, 1662, 1765, 1886, 1908, 1854, 2065, 2261, 2158, 1969, 2079, 2310, 2260, 2083, 1934, 921, 1930, 989, 1910, 1080, 1917, 1139, 1946, 1041, 2063, 2079, 1934, 1808, 2030, 2577, 3238, 3848, 4390, 4933, 5306, 5516, 5427, 5267, 5266, 5583, 5964, 6059, 5714, 4912, 4168, 3847, 3882, 3834, 3562, 3187, 2730, 2201, 1833, 1334, 1831, 2081, 2258, 2311, 2240, 2111, 1912, 1846, 1917, 2017, 2020, 1915, 1750, 1591, 1447, 1395, 1464, 1432, 1417, 1538, 1780, 1948, 1815, 1548, 1464, 1626, 1798, 1856, 1802, 1770, 1755, 1702, 1611, 1476, 1360, 1453, 1613, 1531, 1399, 1502, 1569, 1461, 1417, 1550, 1624, 1561, 1411, 1326, 1383, 1414, 1383, 1394, 1474, 1445, 1372, 1365, 1422, 1605, 1823, 2002, 2049, 1949, 2055, 2341, 1006, 2652, 2786, 2620, 2375, 2276, 2518, 2913, 3236, 3195, 3017, 2988, 2850, 2576, 2460, 2476, 2591, 2642, 2635, 2615, 2486, 2167, 2016, 2180, 2523, 2738, 2709, 2499, 2365, 2084, 1801, 1582, 1533, 1449, 1357, 1283, 1271, 1205, 1229, 1381, 1555, 1513, 1408, 1363, 1471, 1582, 1666, 1639, 1546, 1404, 1457, 1739, 1976, 1953, 1681, 1452, 1276, 1259, 1336, 1350, 1370, 1436, 1545, 1537, 1425, 1295, 1406, 1728, 2047, 2093, 1894, 1775, 1725, 1758, 1870, 1996, 2127, 2248, 2256, 2251, 2125, 1825, 1395, 1204, 1142, 1071, 1037, 1062, 1104, 1169, 1269, 1412, 1489, 1410, 1311, 1324, 1401, 1455, 1529, 1623, 1620, 1392, 1020, 797, 941, 1241, 1440, 1529, 1595, 1621, 1722, 1912, 2122, 2185, 2257, 2599, 3224, 3737, 3817, 3534, 3388, 3425, 3538, 3640, 3702, 3698, 3605, 3375, 3039, 2877, 3033, 3343, 3438, 3409, 3274, 3085, 2753, 2396, 1962, 1727, 1753, 1927, 2019, 1976, 1879, 1693, 1466, 1308, 1232, 1338, 1532, 1620, 1611, 1628, 1625, 1578, 1571, 1726, 1925, 2145, 2374, 2692, 2859, 2838, 2554, 2190, 1784, 1520, 1374, 1168, 585, 913, 419, 715, 596, 597, 639, 644, 568, 492, 438, 438, 462, 505, 519, 476, 413, 361, 410, 637, 960, 1310, 1931, 2962, 4220, 5845, 7231, 8613, 10261, 12400, 14424, 15897, 16552, 17016, 17688, 18416, 18600, 17968, 17480, 17160, 16880, 15919, 15027, 14238, 13668, 13153, 12580, 11380, 9862, 8638, 8055, 7743, 7668, 7691, 7374, 6889, 6189, 5507, 5071, 5059, 5174, 4983, 4565, 4197, 4188, 4155, 3961, 3686, 3511, 3149, 2859, 2786, 2890, 2897, 2685, 2407, 2091, 1833, 1752, 2023, 2321, 2878, 3774, 4799, 5680, 6384, 7086, 7586, 8187, 9046, 9760, 10133, 10272, 10359, 10323, 10261, 10263, 10077, 9661, 9260, 9022, 8918, 8572, 8326, 8302, 8154, 7314, 6433, 6153, 6441, 6512, 6282, 5835, 5433, 5088, 4836, 4432, 3814, 3087, 2479, 2064, 1784, 1547, 1308, 1156, 1038, 984, 1072, 1343, 1509, 1508, 1412, 1397, 1500, 1795, 2399, 3295, 4093, 4715, 5121, 5267, 5204, 5396, 5890, 6142, 5707, 4705, 3909, 3693, 3929, 3915, 3424, 2622, 1990, 1684, 1504, 1392, 1260, 1266, 1386, 1667, 2174, 3172, 4899, 7086, 9832, 12273, 13070, 11689, 9301, 7022, 5029, 3186, 1827, 947, 495, 275, 96, 158, 126, 152, 156, 107, 94, 100, 85, 70, 76, 95, 91, 73, 99, 148, 124, 222, 152, 236, 156, 95, 74, 100, 72, 73, 71)
  
  expected_rt2    <- c(2548.327, 2549.892, 2557.716, 2559.281, 2560.846, 2562.411, 2568.671, 2628.139, 2629.704, 2631.269, 2637.529, 2639.094, 2640.659, 2732.991, 2748.641, 2750.206, 2751.771, 2753.336, 2825.323, 2826.888, 2828.453, 2830.018, 2911.396, 2912.961, 2914.526, 2920.786, 2922.351, 2923.916, 2928.610, 2930.175, 2931.740, 3038.157, 3074.151, 3099.190, 3324.544, 3326.109, 3390.272, 3520.163, 3521.728, 3531.117, 3532.682, 4387.147, 4388.712)
  expected_mz2    <- c(460.80001831054688, 460.80001831054688, 460.80001831054688, 460.80001831054688, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.80001831054688, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.80001831054688, 460.80001831054688, 460.80001831054688, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.80001831054688, 460.89999389648438, 460.89999389648438, 460.80001831054688, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.89999389648438, 460.8999938964843)
  expected_int2   <- c(262, 289, 337, 449, 465, 405, 485, 497, 580, 663, 739, 679, 514, 357, 277, 446, 614, 679, 511, 474, 435, 455, 731, 929, 1055, 1275, 1156, 924, 1403, 1471, 1487, 574, 558, 448, 438, 336, 207, 802, 1022, 1113, 1145, 83, 80)
  
  expected_signal <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, mz=data.frame(matrix(c(520., 521., 460.8, 460.9), ncol=2, byrow=TRUE)), msLevel=1L, verbose=FALSE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(length(result_extractSignalRawData$messages), 0)
})

test_that('no mz window defined, 2 data.frame rt (all mz range + extend mz)', {
  # Expected signal
  expected_rt1  <- c(3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447,
                    3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3382.447, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012,
                    3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012, 3384.012)
  expected_mz1  <- c(200.1000061, 201, 201.9000092, 202.9000092, 204.1000061, 204.9000092, 206, 206.9000092, 208.1000061, 208.8000031, 210, 211.1000061, 211.9000092, 212.4000092, 213.1000061, 214.1999969, 214.9000092, 216, 217.1000061, 218, 219.1000061, 220, 221, 222.1000061, 223, 224.1000061, 225, 226.3000031, 227.1999969, 228, 229, 230.1000061, 231, 232.1999969, 233.1000061, 234, 235.1000061, 236.1000061, 237, 238.1000061, 239.1000061, 240.1000061, 240.9000092, 242, 243.1000061, 244, 245.1000061, 246.1000061, 247.1999969, 248.1999969, 249.1000061, 250.1000061, 251.1000061, 251.8000031, 253.1999969, 254.1999969, 255.1000061, 256.1000061, 257.1000061, 258.1000061, 259.1000061, 259.8999939, 260.2000122, 261.1000061, 262.1000061, 263.1000061, 264.1000061, 265.1000061, 266.3000183, 267.3999939, 268.1000061, 269.2000122, 270.1000061, 271.1000061, 272.2000122, 273.1000061, 273.8999939, 274.2000122, 275.7000122, 276.2000122, 276.8999939, 277.3000183, 278, 279, 280.3000183, 280.7000122, 282.2000122, 283.1000061, 284.3000183, 285.1000061, 286.1000061, 287.1000061, 288.1000061, 289.1000061, 290, 291, 292.3999939, 294.1000061, 294.8999939, 296.1000061, 298.1000061, 300.2000122, 301.2000122, 302.2000122, 303, 304.2000122, 305, 305.3000183, 305.7000122, 307.2000122, 308.2000122, 309.1000061, 310.3999939, 311.1000061, 312.1000061, 313.1000061, 314.2000122, 315.2000122, 316, 317.1000061, 318.1000061, 319.2000122, 320, 320.3000183, 321.3000183, 322.1000061, 323.1000061, 324, 325.1000061, 326.1000061, 327, 328.2000122, 329.3000183, 330.2000122, 331.3000183, 332.1000061, 333, 334.1000061, 335, 335.8000183, 336.2000122, 337.2000122, 338.1000061, 339.2000122, 340.2000122, 342.1000061, 343.1000061, 344, 345.3000183, 346.1000061, 347.2000122, 348.2000122, 349, 350.1000061, 351.2000122, 352.2000122, 352.8000183, 353.3000183, 354.1000061, 355.2000122, 356.2000122, 357.1000061, 358.1000061, 359.2000122, 360.3000183, 361.1000061, 362.1000061, 363.1000061, 365.1000061, 366.2000122, 367.2000122, 368.3000183, 369.3000183, 370.2000122, 371.1000061, 372, 373.2000122, 374.1000061, 375, 375.3999939, 375.8999939, 377, 378.1000061, 378.5, 379.1000061, 380.1000061, 381.1000061, 382.1000061, 383.1000061, 384.2000122, 385.1000061, 386.1000061, 387.2000122, 388.2000122, 389.1000061, 390.1000061, 391.1000061, 392.2000122, 392.8999939, 394.2000122, 395, 396.1000061, 397, 398.2000122, 399.2000122, 400.2000122, 401.1000061, 402.2000122, 403, 403.5, 404.2000122, 405.2000122, 406.1000061, 407.2000122, 408.1000061, 409.1000061, 410, 410.3000183, 411.1000061, 412, 413.2000122, 414.2000122, 415.2000122, 416.3000183, 417.2000122, 418.1000061, 419, 420, 421.1000061, 422.1000061, 423.1000061, 424.1000061, 425, 427, 427.3000183, 428.1000061, 429, 429.3000183, 430.1000061, 431.3000183, 432.2000122, 433, 433.6000061, 434.2000122, 435, 436.1000061, 437.2000122, 438.1000061, 439.1000061, 440.3000183, 441.2000122, 442.3000183, 443.2000122, 444.1000061, 445.1000061, 446.2000122, 447.1000061, 448.3000183, 448.8999939, 449.3000183, 450.2000122, 451.2000122, 452.2000122, 453.2000122, 454.2000122, 455.1000061, 456.2000122, 457.1000061, 458.2000122, 459, 460, 461.1000061, 462.1000061, 463, 463.3999939, 464.3000183, 465.2000122, 466.1000061, 467.1000061, 468.3000183, 469.1000061, 470, 471.1000061, 472.2000122, 473.1000061, 473.5, 475.2000122, 476.2000122, 477.1000061, 478.2000122, 480.2000122, 481.2000122, 482.1000061, 483.2000122, 484.3000183, 485.1000061, 486.6000061, 487.1000061, 488.2000122, 489.2000122, 491.2000122, 492.2000122, 493.1000061, 494.1000061, 496.2000122, 497.2000122, 498.2000122, 499.2000122, 500.2000122, 501.2000122, 502.1000061, 503.2000122, 504.1000061, 505.2000122, 506.2000122, 507.2000122, 508.1000061, 509.1000061, 510.2000122, 511.1000061, 512.1000366, 513.1000366, 514.2999878, 515.2999878, 515.7999878, 516.9000244, 518.2000122, 519.2000122, 520.1000366, 522.2000122, 523.2000122, 524, 525.1000366, 526.1000366, 527.1000366, 528.1000366,
                    529.1000366, 530.2000122, 531.1000366, 532.2000122, 533.2000122, 534.1000366, 535.1000366, 536.1000366, 537.2999878, 538.2000122, 539.2999878, 540.2999878, 541.2000122, 542, 543.1000366, 544.2000122, 545.2999878, 546.2000122, 547.2000122, 548.1000366, 549, 549.4000244, 550.2999878, 551.1000366, 552.2000122, 553.1000366, 554.2999878, 555.2999878, 556.1000366, 557.2000122, 558.1000366, 559.2000122, 560.2000122, 561.2000122, 562.2000122, 563.1000366, 564.2000122, 565.2000122, 566.2000122, 567.4000244, 568.1000366, 569, 570, 572.2000122, 573.2000122, 574.2999878, 575.2000122, 576.2999878, 577.1000366, 578.2000122, 579, 579.2999878, 580.2999878, 581.2000122, 582.2000122, 583, 583.2999878, 584.2000122, 585.4000244, 586.2000122, 587.2000122, 588.2999878, 589, 590.1000366, 591.2999878, 592.2000122, 593.2000122, 594.2000122, 595.4000244, 596.1000366, 597.1000366, 598.1000366, 599.2999878, 200.1000061, 201, 201.9000092, 202.9000092, 204.1000061, 205, 206, 207, 208.1000061, 208.8000031, 209.1000061, 209.9000092, 211.1000061, 212, 213.1000061, 214.1999969, 215, 216.1000061, 217, 217.9000092, 219.1000061, 220, 220.9000092, 221.9000092, 223, 224.1000061, 225.1000061, 226.1999969, 227.1999969, 228.1000061, 228.9000092, 230.1999969, 231, 232.1000061, 233.1000061, 233.9000092, 235.1000061, 236.1000061, 237, 238, 239.1000061, 240.1000061, 241, 242, 243.1000061, 244, 245.1000061, 246, 247.3000031, 248.1999969, 249.1000061, 250.1000061, 251.1999969, 251.8000031, 253.1999969, 254.1999969, 255.1000061, 256.2000122, 257.1000061, 258.1000061, 259.2000122, 259.8999939, 260.3000183, 261.1000061, 262.1000061, 263.1000061, 263.8000183, 265.1000061, 266.3000183, 267.3999939, 268.1000061, 269.1000061, 270.1000061, 271.1000061, 272.3000183, 273.1000061, 274.1000061, 275.7000122, 276.2000122, 276.8000183, 277.3999939, 278, 279, 280.2000122, 280.8000183, 282.1000061, 283.1000061, 284.3000183, 285.1000061, 286.1000061, 287.2000122, 288.1000061, 289.1000061, 290, 291.2000122, 292.7000122, 294.1000061, 294.8999939, 295.2000122, 296, 298.1000061, 300.2000122, 301.2000122, 302.2000122, 303, 304.1000061, 305, 305.7000122, 307.2000122, 308.2000122, 309.1000061, 310.2000122, 311.1000061, 312, 313.1000061, 314.2000122, 315.2000122, 316, 317.1000061, 318.1000061, 319.1000061, 320.3000183, 321.3000183, 322.1000061, 323.2000122, 324.1000061, 325.1000061, 326.1000061, 327, 328.2000122, 329.2000122, 330.2000122, 331.3000183, 332.1000061, 333, 334.1000061, 335, 335.8000183, 336.2000122, 336.8000183, 337.2000122, 338.1000061, 339.2000122, 340.2000122, 342.1000061, 343.1000061, 344, 345.3000183, 346.2000122, 347.1000061, 348.2000122, 349, 350.1000061, 351.1000061, 352.2000122, 353.2000122, 354.1000061, 355.2000122, 356.2000122, 357.1000061, 358, 359.1000061, 360.3000183, 361.1000061, 362.1000061, 363.1000061, 364.1000061, 365.2000122, 366.1000061, 367.3000183, 368.2000122, 369.3000183, 370.2000122, 371.1000061, 372.1000061, 373.3000183, 374.2000122, 375, 375.3999939, 375.8999939, 377, 378, 378.5, 379.1000061, 380.1000061, 381.1000061, 382.1000061, 383.2000122, 384.3000183, 385.1000061, 386.1000061, 387.1000061, 388.2000122, 389.1000061, 390.1000061, 391.2000122, 392.2000122, 393, 394.1000061, 395, 396.1000061, 397, 398.2000122, 399.2000122, 400.2000122, 401.2000122, 402.2000122, 403.5, 404.1000061, 405.2000122, 406, 406.3000183, 407.2000122, 408.1000061, 409.2000122, 410, 410.3999939, 411.1000061, 412, 413.1000061, 414, 415.2000122, 416.3000183, 417.3000183, 418.1000061, 419, 420.1000061, 421.1000061, 422.2000122, 423.1000061, 424.1000061, 424.3999939, 425, 427.1000061, 428.1000061,
                    429, 429.3999939, 430.1000061, 431.3999939, 432.2000122, 433, 433.6000061, 434.2000122, 435, 436.1000061, 437.2000122, 438.1000061, 439.2000122, 440, 441.3000183, 442.2000122, 443.2000122, 444.1000061, 445.1000061, 446.1000061, 447.1000061, 448.3000183, 449.2000122, 450.2000122, 451.2000122, 452.2000122, 453.2000122, 454.1000061, 455.1000061, 456.1000061, 457.1000061, 458.2000122, 459, 459.3000183, 460.1000061, 461.1000061, 462.1000061, 463, 463.3000183, 464.3000183, 465.2000122, 466.2000122, 467, 468.3000183, 469.1000061, 470, 471.2000122, 472.3000183, 473, 475.2000122, 476.2000122, 477.2000122, 478.2000122, 480.2000122, 481.1000061, 482.1000061, 483.2000122, 484, 484.3000183, 485.2000122, 486.5, 487.1000061, 488.2000122, 489.3000183, 491.2000122, 492.2000122, 493.1000061, 494.1000061, 496.2000122, 497.2000122, 498.2000122, 499.1000061, 500.2000122, 501.2000122, 502.1000061, 503.2000122, 504.1000061, 506.2000122, 507.2000122, 508.1000061, 509.1000061, 510.2000122, 511.2000122, 512.1000366, 513.1000366, 514.2999878, 515.2999878, 515.7999878, 518.2000122, 519.1000366, 520.1000366, 522.2000122, 523.2000122, 524, 525, 526.1000366, 527.1000366, 528.1000366, 529.1000366, 530.1000366, 531.2000122, 532.2000122, 533.2000122, 534.1000366, 535.1000366, 536.1000366, 537.2999878, 538.1000366, 539.2999878, 540.2999878, 541.2000122, 542, 543.1000366, 544.2000122, 545.1000366, 546.2000122, 547.2000122, 548.2000122, 549.4000244, 550.1000366, 551.2000122, 552.2000122, 553.2000122, 554.2999878, 555.2000122, 556.2000122, 557.2000122, 558.1000366, 559.2000122, 560.2000122, 561.2000122, 562.2000122, 563.2000122, 564.1000366, 565.2000122, 566.2000122, 567.2999878, 568.1000366, 569.1000366, 570, 570.9000244, 572.2000122, 573.2999878, 574.2000122, 575.2999878, 576.2000122, 577.2000122, 578.2000122, 579, 580.2000122, 581.2000122, 582.2000122, 582.9000244, 583.4000244, 584.2000122, 585.4000244, 586.1000366, 587.2000122, 588.2999878, 589.1000366, 590.2000122, 591.2000122, 592.2000122, 593.2000122, 594.2000122, 595.4000244, 596.1000366, 597.1000366, 598.1000366, 599.2999878)
  expected_int1 <- c(729, 439, 1702, 336, 401, 485, 277, 970, 201, 335, 357, 860, 450, 195, 635, 348, 464, 768, 744, 121, 426, 226, 2330, 365, 786, 642, 637, 255, 1883, 731, 629, 1507, 3615, 645, 671, 344, 358, 372, 282, 248, 1660, 681, 1056, 739, 321, 635, 547, 160, 338, 413, 855, 1018, 486, 304, 789, 484, 513, 947, 939, 583, 308, 281, 269, 524, 259, 266, 201, 536, 331, 1893, 948, 1206, 821, 1271, 870, 861, 248, 224, 2724, 989, 737, 493, 199, 1572, 1725, 1656, 1928, 1276, 2455, 1054, 352, 1192, 1513, 1087, 354, 493, 1096, 333, 560, 3824, 1416, 201152, 42632, 4976, 1449, 2452, 983, 991, 1301, 590, 667, 1289, 426, 1362, 906, 1973, 1381, 630, 538, 645, 425, 316, 514, 502, 644, 9931, 2057, 640, 383, 582, 717, 2580, 754, 717, 663, 825, 803, 826, 482, 256, 348, 474, 1122, 7811, 7828, 939, 768, 772, 1854, 844, 655, 455, 1004, 876, 621, 504, 138, 204, 1224, 722, 460, 769, 777, 1339, 860, 1452, 582, 643, 10111, 2953, 1387, 2629, 1867, 1851, 1550, 827, 890, 363, 436, 256, 424, 325, 248, 187, 593, 2203, 895, 1858, 1591, 769, 988, 566, 1302, 903, 4637, 1352, 1253, 396, 820, 369, 530, 1770, 749, 514, 3364, 2054, 1560, 785, 351, 508, 644, 515, 313, 319, 426, 786, 274, 418, 1120, 877, 619, 642, 561, 780, 1190, 1347, 901, 437, 764, 460, 622, 720, 3104, 592, 586, 703, 592, 559, 808, 450, 639, 468, 337, 280, 377, 360, 678, 848, 555, 301, 614, 885, 692, 465, 602, 268, 545, 306, 208, 299, 350, 358, 1035, 16784, 10860, 2714, 815, 382, 568, 620, 592, 364, 923, 576, 530, 1040, 689, 665, 406, 496, 584, 320, 502, 373, 410, 227, 18376, 5992, 1292, 2784, 98336, 25144, 5427, 1071, 606, 364, 379, 563, 1226, 837, 3486, 1766, 687, 642, 1095680, 310912, 56272, 8452, 1443, 1379, 10838, 3000, 4372, 1650, 136896, 42832, 11282, 3295, 1487, 498, 1626, 702, 633, 4594, 2356, 501, 33632, 10312, 3504, 53120, 26624, 8209, 1664, 9528, 2928, 10254, 3679, 70288, 22408, 5163, 1530, 4178, 1508, 2253, 762, 1098, 855, 6211, 2826, 1175, 574, 2444, 1629, 1944, 596, 1027, 331, 384, 950, 640, 4352, 1618, 2793, 1042, 906, 425, 4187, 1580, 1943, 862, 594, 341, 1202, 801, 1421, 719, 1124, 873, 754, 10283, 3287, 1153, 821, 750, 525, 891, 398, 355, 2887, 1089, 632, 426, 406, 718, 568, 683, 289, 400, 488, 680, 430, 516, 664, 2053, 1199, 4192, 1369, 1075, 1867, 593, 412, 1620, 353, 364, 414, 298, 832, 223, 284, 272, 340, 830, 459, 624, 331, 411, 714, 730, 157, 431, 208, 2227, 404, 890, 710, 554, 278, 1873, 761, 632, 1441, 3498, 601, 619, 440, 369, 315, 300, 271, 1704, 667, 1039, 786, 361, 568, 667, 214, 314, 501, 739, 916, 433, 195, 886, 462, 498, 1013, 768, 670, 351, 271, 209, 462, 226, 349, 281, 438, 276, 1799, 995, 1201, 885, 1269, 771, 846, 295, 2847, 1093, 749, 294, 242, 1423, 1574, 1386, 1907, 1514, 2086, 953, 373, 1251, 1386, 1140, 364, 482, 997, 306, 487, 388, 3771, 1305, 183040, 39768, 4274, 1362, 2162, 971, 1211, 584, 629, 1180, 504, 1590, 824, 1976, 1340, 646, 544, 660, 512, 396, 565, 554, 9214, 1800, 639, 356, 647, 768, 2091, 639, 629, 597, 826, 677, 766, 621, 286, 378, 418, 376, 918, 8212, 7574, 854, 877, 899, 1961, 804, 708, 509, 1110, 782, 472, 436, 260, 1235, 805, 539, 735, 742, 1325, 804, 1373, 654, 563, 598, 10081, 2892, 1596, 2834, 1808, 1809, 1547, 1026, 927, 363, 478, 343, 382, 306, 254, 202, 530, 2099, 878, 1846, 1451, 793, 1164, 558, 1305, 845, 4508, 1484, 1181, 453, 731, 583, 521, 1832, 788, 560, 3266, 1955, 1530, 784, 662, 726, 609, 322, 342, 324,
                    452, 849, 278, 394, 1001, 776, 716, 634, 502, 696, 1110, 1286, 866, 532, 840, 499, 716, 539, 499, 2874, 697, 759, 744, 567, 752, 401, 563, 416, 365, 215, 308, 462, 711, 759, 466, 390, 528, 816, 654, 406, 653, 308, 526, 294, 295, 303, 509, 998, 16381, 10824, 2434, 881, 311, 573, 590, 493, 644, 394, 914, 498, 550, 923, 725, 682, 406, 565, 512, 414, 480, 392, 352, 18320, 6069, 1489, 2790, 97168, 24592, 5232, 1073, 495, 503, 346, 447, 588, 1060, 809, 3383, 1726, 609, 905, 1122304, 309056, 56856, 8057, 1421, 1327, 10636, 2971, 4367, 135424, 42168, 11149, 3218, 1357, 616, 1611, 715, 679, 4437, 2469, 33624, 10665, 3751, 62488, 29248, 8731, 1757, 9592, 2897, 10366, 3907, 65136, 20768, 5098, 1519, 3938, 1463, 2584, 915, 1225, 952, 6137, 2921, 1190, 613, 2975, 1479, 1978, 721, 1062, 368, 931, 618, 4938, 2003, 2866, 1094, 824, 500, 3971, 1456, 1885, 805, 643, 295, 1148, 832, 1378, 735, 982, 744, 823, 178, 9144, 2934, 1243, 673, 977, 544, 952, 475, 2948, 1026, 620, 497, 402, 813, 604, 627, 297, 349, 514, 605, 474, 493, 755, 1648, 1037, 4346, 1299, 917, 1685)
  
  expected_rt2    <- c(3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272,
                       3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272, 3390.272)
  expected_mz2    <- c(200.90000915527344, 201.90000915527344, 203.00000000000000, 204.00000000000000, 204.90000915527344, 206.00000000000000, 207.00000000000000, 208.00000000000000, 209.10000610351562, 210.19999694824219, 211.10000610351562, 212.00000000000000, 213.10000610351562, 213.80000305175781, 214.19999694824219, 215.10000610351562, 215.80000305175781, 216.19999694824219, 217.00000000000000, 218.19999694824219, 218.90000915527344, 219.90000915527344, 221.00000000000000, 222.00000000000000, 223.00000000000000, 224.10000610351562, 225.00000000000000, 226.00000000000000, 227.10000610351562, 228.00000000000000, 229.00000000000000, 230.19999694824219, 231.00000000000000, 232.00000000000000, 232.69999694824219, 233.10000610351562, 233.90000915527344, 235.00000000000000, 236.00000000000000, 237.00000000000000, 238.00000000000000, 239.00000000000000, 240.00000000000000, 241.10000610351562, 242.00000000000000, 243.00000000000000, 244.00000000000000, 245.10000610351562, 246.10000610351562, 247.00000000000000, 248.00000000000000, 249.00000000000000, 250.10000610351562, 251.10000610351562, 252.19999694824219, 253.00000000000000, 254.00000000000000, 255.00000000000000, 256.00000000000000, 257.20001220703125, 258.00000000000000, 259.20001220703125, 259.60000610351562, 261.20001220703125, 261.89999389648438, 263.20001220703125, 264.10000610351562, 265.00000000000000, 266.10000610351562, 267.70001220703125, 269.10000610351562, 270.00000000000000, 271.20001220703125, 272.20001220703125, 273.00000000000000, 274.30001831054688, 274.80001831054688, 275.60000610351562, 276.30001831054688, 277.00000000000000, 277.89999389648438, 278.89999389648438, 280.30001831054688, 280.80001831054688, 282.20001220703125, 283.20001220703125, 284.30001831054688, 285.00000000000000, 286.20001220703125, 287.10000610351562, 288.39999389648438, 288.89999389648438, 290.30001831054688, 291.20001220703125, 291.70001220703125, 292.60000610351562, 293.10000610351562, 294.10000610351562, 295.00000000000000, 296.20001220703125, 297.00000000000000, 298.10000610351562, 299.10000610351562, 300.20001220703125, 301.20001220703125, 302.10000610351562, 303.10000610351562, 304.10000610351562, 305.39999389648438, 306.20001220703125, 306.80001831054688, 307.30001831054688, 308.10000610351562, 309.10000610351562, 310.10000610351562, 311.00000000000000, 311.89999389648438, 313.10000610351562, 314.10000610351562, 315.00000000000000, 316.00000000000000, 317.20001220703125, 318.30001831054688, 319.20001220703125, 320.20001220703125, 321.10000610351562, 322.10000610351562, 323.20001220703125, 324.10000610351562, 325.10000610351562, 326.20001220703125, 327.20001220703125, 328.20001220703125, 329.00000000000000, 329.30001831054688, 330.00000000000000, 331.10000610351562, 332.10000610351562, 333.10000610351562, 334.20001220703125, 335.20001220703125, 336.10000610351562, 337.20001220703125, 338.10000610351562, 339.10000610351562, 340.10000610351562, 341.20001220703125, 341.89999389648438, 342.30001831054688, 343.20001220703125, 344.00000000000000, 345.20001220703125, 346.10000610351562, 347.30001831054688, 348.10000610351562, 349.10000610351562, 350.00000000000000, 351.20001220703125, 352.10000610351562, 353.10000610351562, 354.30001831054688, 355.20001220703125, 356.00000000000000, 357.20001220703125, 358.10000610351562, 359.10000610351562, 360.10000610351562, 361.10000610351562, 362.10000610351562, 363.10000610351562, 364.10000610351562, 365.20001220703125, 366.10000610351562, 367.10000610351562, 368.20001220703125, 369.20001220703125, 370.10000610351562, 371.10000610351562, 372.20001220703125, 373.20001220703125, 374.20001220703125, 375.10000610351562,
                       376.20001220703125, 377.00000000000000, 377.60000610351562, 378.10000610351562, 379.20001220703125, 380.00000000000000, 381.10000610351562, 382.20001220703125, 383.00000000000000, 384.10000610351562, 385.10000610351562, 386.10000610351562, 386.39999389648438, 387.10000610351562, 388.20001220703125, 389.10000610351562, 390.10000610351562, 391.20001220703125, 392.20001220703125, 393.10000610351562, 394.30001831054688, 395.10000610351562, 396.10000610351562, 397.30001831054688, 398.20001220703125, 399.20001220703125, 400.10000610351562, 401.20001220703125, 402.30001831054688, 403.39999389648438, 404.20001220703125, 405.10000610351562, 405.80001831054688, 406.30001831054688, 407.20001220703125, 407.89999389648438, 409.10000610351562, 410.20001220703125, 411.10000610351562, 412.20001220703125, 413.00000000000000, 414.10000610351562, 415.20001220703125, 416.00000000000000, 416.39999389648438, 417.30001831054688, 418.10000610351562, 419.10000610351562, 420.00000000000000, 421.00000000000000, 422.30001831054688, 423.10000610351562, 424.00000000000000, 425.00000000000000, 426.20001220703125, 426.89999389648438, 428.20001220703125, 429.30001831054688, 430.10000610351562, 431.00000000000000, 432.30001831054688, 433.30001831054688, 433.89999389648438, 435.30001831054688, 436.00000000000000, 436.30001831054688, 437.10000610351562, 438.20001220703125, 439.20001220703125, 440.00000000000000, 440.30001831054688, 440.89999389648438, 441.30001831054688, 442.00000000000000, 443.00000000000000, 444.30001831054688, 445.00000000000000, 446.10000610351562, 447.20001220703125, 448.00000000000000, 449.20001220703125, 450.10000610351562, 451.10000610351562, 452.10000610351562, 453.20001220703125, 454.20001220703125, 455.20001220703125, 456.20001220703125, 457.30001831054688, 458.10000610351562, 459.00000000000000, 460.10000610351562, 460.80001831054688, 461.30001831054688, 462.39999389648438, 463.10000610351562, 464.30001831054688, 465.20001220703125, 465.89999389648438, 466.39999389648438, 467.20001220703125, 468.20001220703125, 469.30001831054688, 470.20001220703125, 471.00000000000000, 472.00000000000000, 473.20001220703125, 475.20001220703125, 476.10000610351562, 477.10000610351562, 478.20001220703125, 479.10000610351562, 480.20001220703125, 481.10000610351562, 482.10000610351562, 483.10000610351562, 484.10000610351562, 485.30001831054688, 486.00000000000000, 486.50000000000000, 487.20001220703125, 488.20001220703125, 489.10000610351562, 490.30001831054688, 491.10000610351562, 492.10000610351562, 493.00000000000000, 493.39999389648438, 494.10000610351562, 496.20001220703125, 497.20001220703125, 498.20001220703125, 499.20001220703125, 500.10000610351562, 501.30001831054688, 502.20001220703125, 503.10000610351562, 504.10000610351562,
                       505.20001220703125, 506.20001220703125, 507.20001220703125, 508.10000610351562, 509.20001220703125, 510.10000610351562, 511.20001220703125, 512.20001220703125, 513.20001220703125, 514.10003662109375, 515.20001220703125, 515.90002441406250, 517.00000000000000, 518.10003662109375, 519.10003662109375, 520.20001220703125, 522.20001220703125, 523.20001220703125, 524.10003662109375, 525.10003662109375, 526.10003662109375, 527.10003662109375, 528.20001220703125, 529.10003662109375, 530.10003662109375, 531.20001220703125, 532.10003662109375, 533.10003662109375, 534.10003662109375, 535.20001220703125, 536.20001220703125, 538.00000000000000, 539.20001220703125, 540.40002441406250, 541.29998779296875, 542.10003662109375, 543.29998779296875, 544.20001220703125, 545.10003662109375, 546.10003662109375, 547.20001220703125, 548.10003662109375, 549.10003662109375, 550.29998779296875, 551.29998779296875, 552.20001220703125, 553.10003662109375, 554.29998779296875, 555.29998779296875, 556.10003662109375, 558.20001220703125, 559.20001220703125, 560.10003662109375, 561.10003662109375, 562.20001220703125, 563.00000000000000, 564.20001220703125, 565.00000000000000, 565.40002441406250, 566.29998779296875, 567.10003662109375, 568.00000000000000, 568.40002441406250, 569.20001220703125, 570.20001220703125, 571.10003662109375, 572.20001220703125, 573.20001220703125, 574.10003662109375, 575.20001220703125, 576.20001220703125, 577.29998779296875, 578.20001220703125, 579.10003662109375, 580.20001220703125, 581.20001220703125, 582.00000000000000, 583.10003662109375, 584.10003662109375, 585.29998779296875, 586.20001220703125, 587.00000000000000, 588.10003662109375, 588.90002441406250, 589.29998779296875, 590.00000000000000, 590.90002441406250, 591.29998779296875, 592.40002441406250, 593.20001220703125, 594.20001220703125, 595.10003662109375, 596.10003662109375, 597.20001220703125, 598.20001220703125, 599.40002441406250)
  expected_int2   <- c(413, 1138, 290, 147, 314, 298, 773, 260, 465, 385, 783, 514, 549, 239, 455, 564, 372, 406, 937, 250, 630, 244, 2350, 484, 870, 553, 1020, 254, 1447, 837, 748, 1323, 3960, 1056, 366, 436, 194, 425, 288, 252, 417, 1664, 769, 1732, 621, 302, 364, 438, 338, 335, 275, 649, 1021, 510, 279, 1027, 480, 623, 935, 1218, 670, 303, 480, 430, 259, 395, 334, 465, 258, 1797, 1605, 939, 1685, 1106, 918, 407, 488, 2464, 1090, 423, 339, 1839, 2031, 1507, 1296, 1346, 2108, 1128, 504, 1338, 1909, 1532, 554, 510, 517, 636, 556, 380, 429, 2392, 1047, 998, 1197, 115192, 25152, 2796, 1169, 2058, 2246, 964, 474, 364, 641, 986, 817, 2034, 1068, 2126, 1439, 953, 613, 608, 537, 419, 472, 411, 6173, 1518, 671, 494, 6194, 1725, 1389, 371, 293, 607, 476, 790, 575, 544, 512, 269, 495, 934, 6658, 7991, 1249, 419, 477, 1190, 838, 1535, 519, 482, 779, 727, 476, 442, 257, 384, 1073, 691, 516, 722, 796, 1044, 656, 1057, 435, 401, 489, 10571, 3343, 2963, 3294, 2342, 1952, 1654, 995, 933, 455, 244, 366, 262, 192, 330, 549, 1849, 688, 1976, 1391, 614, 1173, 394, 308, 960, 734, 3764, 1168, 1477, 451, 829, 399, 582, 1436, 821, 452, 3274, 1717, 1772, 778, 478, 482, 349, 133, 172, 394, 539, 516, 436, 860, 753, 479, 653, 841, 366, 415, 1220, 1287, 711, 774, 900, 483, 880, 737, 1937, 525, 667, 1012, 922, 674, 419, 578, 406, 400, 337, 503, 412, 734, 1263, 671, 780, 840, 581, 447, 855, 410, 543, 826, 482, 1276, 498, 403, 329, 386, 1370, 17288, 11711, 2267, 936, 817, 542, 676, 306, 207, 508, 700, 677, 697, 1474, 506, 771, 374, 503, 667, 487, 432, 293, 533, 19264, 5941, 1268, 2913, 1891, 89544, 22736, 5293, 943, 568, 502, 220, 327, 755, 1344, 1097, 486, 2895, 1341, 398, 442, 1054, 996608, 271872, 52296, 8018, 1559, 1620, 11008, 2874, 4309, 1183, 141440, 40488, 12308, 3918, 1398, 564, 1752, 702, 552, 3651, 1636, 424, 31272, 8436, 3541, 173440, 61272, 13455, 2723, 9388, 3312, 10617, 3633, 56792, 18824, 4470, 2040, 4218, 1553, 3954, 1297, 831, 4986, 3056, 1148, 549, 6189, 3150, 1754, 1014, 903, 879, 1376, 873, 10108, 3262, 3289, 1388, 925, 3652, 1503, 1926, 874, 690, 444, 954, 457, 383, 1740, 1074, 960, 975, 688, 545, 346, 5890, 1796, 1281, 497, 1264, 653, 870, 470, 3681, 1505, 664, 302, 1202, 380, 674, 355, 464, 246, 328, 677, 492, 538, 540, 673, 1530, 542, 3363, 1515, 648, 1079)
  expected_signal <- list(data.frame(rt=expected_rt1, mz=expected_mz1, int=expected_int1), data.frame(rt=expected_rt2, mz=expected_mz2, int=expected_int2))
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3382., 3385., 3390., 3390.5), ncol=2, byrow=TRUE), msLevel=1L, verbose=FALSE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(length(result_extractSignalRawData$messages), 0)
})

test_that('set msLevel (warning no msLevel', {
  # Expected
  expected_signal   <- list(data.frame(rt=numeric(), mz=numeric(), int=integer()))
  expected_messages <- c("Reading data from 1 windows\n", "No data exist for MS level 2\n")
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, msLevel=2L, verbose=TRUE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check message
  expect_equal(result_extractSignalRawData$messages, expected_messages)
})

test_that('no rt found across all windows (message)', {
  # expected
  expected_signal   <- list(data.frame(rt=numeric(), mz=numeric(), int=integer()))
  expected_messages <- c('Reading data from 1 windows\n', 'No data exist for the rt provided\n')
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=c(3310., 3310.), mz=c(520., 520.), msLevel=1L, verbose=TRUE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages
  expect_equal(result_extractSignalRawData$messages, expected_messages)
})

test_that('no data for 1 window (no scans in rt range)', {
  # expected
  expected_rt  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  expected_signal   <- list(data.frame(rt=expected_rt, mz=expected_mz, int=expected_int), data.frame(rt=numeric(), mz=numeric(), int=integer()))
  expected_messages <- c('Reading data from 2 windows\n', 'No data exist for window 2\n')
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3310., 3390., 3310., 3310.), ncol=2, byrow=TRUE), mz=matrix(c(520., 522.4, 520., 522.4), ncol=2, byrow=TRUE), msLevel=1L, verbose=TRUE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages (skip timing)
  expect_equal(length(result_extractSignalRawData$messages), 3)
  expect_equal(result_extractSignalRawData$messages[1:2], expected_messages)
})

test_that('no data for 1 window (no scans in mz range)', {
  # expected
  expected_rt  <- c(3310.459, 3310.459, 3310.459, 3312.024, 3312.024, 3312.024, 3313.589, 3313.589, 3313.589, 3315.154, 3315.154, 3315.154, 3316.719, 3316.719, 3316.719, 3318.284, 3318.284, 3318.284, 3319.849, 3319.849, 3319.849, 3321.414, 3321.414, 3321.414, 3322.979, 3322.979, 3322.979, 3324.544, 3324.544, 3324.544, 3326.109, 3326.109, 3327.673, 3327.673, 3329.238, 3329.238, 3329.238, 3330.803, 3330.803, 3332.368, 3332.368, 3333.933, 3333.933, 3335.498, 3335.498, 3337.063, 3337.063, 3338.628, 3338.628, 3340.193, 3340.193, 3341.758, 3341.758, 3343.323, 3343.323, 3344.888, 3344.888, 3346.453, 3346.453, 3348.018, 3348.018, 3348.018, 3349.583, 3349.583, 3349.583, 3351.148, 3351.148, 3351.148, 3352.713, 3352.713, 3352.713, 3354.278, 3354.278, 3355.843, 3355.843, 3357.408, 3357.408, 3358.973, 3358.973, 3360.538, 3360.538, 3362.102, 3362.102, 3363.667, 3363.667, 3365.232, 3365.232, 3366.797, 3366.797, 3368.362, 3368.362, 3369.927, 3369.927, 3369.927, 3371.492, 3371.492, 3371.492, 3373.057, 3373.057, 3373.057, 3374.622, 3374.622, 3374.622, 3376.187, 3376.187, 3376.187, 3377.752, 3377.752, 3377.752, 3379.317, 3379.317, 3380.882, 3380.882, 3382.447, 3382.447, 3384.012, 3384.012, 3385.577, 3385.577, 3385.577, 3387.142, 3387.142, 3387.142, 3388.707, 3388.707, 3388.707)
  expected_mz  <- c(520.2000122, 521.2999878, 522, 520.2000122, 521, 522.1000366, 520.2000122, 521, 522.1000366, 520.2000122, 521.1000366, 522.2000122, 520.2000122, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2999878, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.4000244, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 520.2999878, 522.2000122, 520, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.2999878, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 520.9000244, 522.2000122, 520, 521, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.1000366, 521.1000366, 522.2000122, 520.2000122, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.1000366, 521.2000122, 522.2000122, 520.2000122, 521.2000122, 522.2000122)
  expected_int <- c(1850, 425, 879, 1864, 408, 1112, 1637, 493, 1540, 1466, 590, 2187, 1452, 668, 3534, 1655, 653, 6338, 1828, 665, 11718, 2034, 712, 21744, 2225, 747, 37872, 2485, 814, 62424, 2526, 98408, 2407, 152896, 2186, 925, 225984, 2033, 308672, 1975, 399360, 1997, 504000, 1853, 614656, 1629, 711872, 1420, 784704, 1311, 836608, 1322, 866304, 1360, 882304, 1261, 889280, 1061, 888256, 936, 902, 866816, 958, 920, 827392, 1178, 997, 777728, 1366, 1129, 727040, 1416, 678464, 1541, 629120, 1621, 578048, 1499, 524288, 1300, 471040, 1619, 416320, 2171, 360064, 2632, 302400, 3164, 249152, 3686, 202560, 3914, 1755, 161024, 3679, 1560, 123520, 3255, 1329, 93160, 3230, 1172, 71856, 3475, 1146, 58392, 3573, 1206, 51072, 3427, 48376, 3360, 49168, 3504, 53120, 3751, 62488, 3761, 1061, 78680, 3562, 1336, 102840, 3472, 1338, 134656)
  expected_signal   <- list(data.frame(rt=expected_rt, mz=expected_mz, int=expected_int), data.frame(rt=numeric(), mz=numeric(), int=integer()))
  expected_messages <- c('Reading data from 2 windows\n')
  
  # results (output, warnings and messages)
  result_extractSignalRawData <- evaluate_promise(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3310., 3390., 3310., 3390.), ncol=2, byrow=TRUE), mz=matrix(c(520., 522.4, 521.1, 521.1), ncol=2, byrow=TRUE), msLevel=1L, verbose=TRUE))
  
  # Check results
  expect_equal(result_extractSignalRawData$result, expected_signal)
  
  # Check result messages (skip timing)
  expect_equal(length(result_extractSignalRawData$messages), 2)
  expect_equal(result_extractSignalRawData$messages[1], expected_messages)
})

test_that('raise errors', {
  # msLevel is not integer
  expect_error(extractSignalRawData(rawSpec=raw_data, msLevel='notInteger'), 'Check input "msLevel" must be integer', fixed=TRUE)
  
  # rt is not numeric, matrix or data.frame
  expect_error(extractSignalRawData(rawSpec=raw_data, rt='not a numeric'), 'Check input "rt" must be numeric, matrix or data.frame', fixed=TRUE)
  # rt is not of length 2
  expect_error(extractSignalRawData(rawSpec=raw_data, rt=3380.), 'Check input "rt" must be numeric of length 2', fixed=TRUE)
  # rt is not a matrix or dataframe with 2 columns
  expect_error(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3380.), ncol=1)), 'Check input "rt" must be a matrix or data.frame with 2 columns', fixed=TRUE)
  expect_error(extractSignalRawData(rawSpec=raw_data, rt=data.frame(matrix(c(3380.), ncol=1))), 'Check input "rt" must be a matrix or data.frame with 2 columns', fixed=TRUE)
  
  # mz is not numeric, matrix or data.frame
  expect_error(extractSignalRawData(rawSpec=raw_data, mz='not a numeric'), 'Check input "mz" must be numeric, matrix or data.frame', fixed=TRUE)
  # mz is not of length 2
  expect_error(extractSignalRawData(rawSpec=raw_data, mz=520.), 'Check input "mz" must be numeric of length 2', fixed=TRUE)
  # mz is not a matrix or dataframe with 2 columns
  expect_error(extractSignalRawData(rawSpec=raw_data, mz=matrix(c(520.), ncol=1)), 'Check input "mz" must be a matrix or data.frame with 2 columns', fixed=TRUE)
  expect_error(extractSignalRawData(rawSpec=raw_data, mz=data.frame(matrix(c(520.), ncol=1))), 'Check input "mz" must be a matrix or data.frame with 2 columns', fixed=TRUE)
  
  # rt and mz have different number of rows
  expect_error(extractSignalRawData(rawSpec=raw_data, rt=matrix(c(3380.,3381.,3390.,3391., 3392.,3393.), ncol=2, byrow=TRUE), mz=matrix(c(520.,521., 521.,522.), ncol=2, byrow=TRUE)), 'Check input "rt" and "mz" matrix or data.frame must have the same number of rows', fixed=TRUE)
  expect_error(extractSignalRawData(rawSpec=raw_data, rt=data.frame(matrix(c(3380.,3381.,3390.,3391., 3392.,3393.), ncol=2, byrow=TRUE)), mz=data.frame(matrix(c(520.,521., 521.,522.), ncol=2, byrow=TRUE))), 'Check input "rt" and "mz" matrix or data.frame must have the same number of rows', fixed=TRUE)
})


