% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrix-based.R
\name{score_transcripts}
\alias{score_transcripts}
\title{Scores transcripts with position weight matrices}
\usage{
score_transcripts(
  sequences,
  motifs = NULL,
  max_hits = 5,
  threshold_method = c("p_value", "relative"),
  threshold_value = 0.25^6,
  n_cores = 1,
  cache = paste0(tempdir(), "/sc/")
)
}
\arguments{
\item{sequences}{character vector of named sequences
(only containing upper case characters A, C, G, T), where the names are
RefSeq identifiers
and sequence
type qualifiers (\code{"3UTR"}, \code{"5UTR"}, \code{"mRNA"}), e.g.
\code{"NM_010356|3UTR"}}

\item{motifs}{a list of motifs that is used to score the specified sequences.
If \code{is.null(motifs)} then all Transite motifs are used.}

\item{max_hits}{maximum number of putative binding sites per mRNA
that are counted}

\item{threshold_method}{either \code{"p_value"} (default) or
\code{"relative"}.
If \code{threshold_method} equals \code{"p_value"}, the default
\code{threshold_value}
 is \code{0.25^6}, which is
lowest p-value that can be achieved by hexamer motifs, the shortest
supported motifs.
If \code{threshold_method} equals \code{"relative"}, the default
\code{threshold_value}
is \code{0.9}, which is 90\% of the maximum PWM score.}

\item{threshold_value}{semantics of the \code{threshold_value} depend on
\code{threshold_method} (default is 0.25^6)}

\item{n_cores}{the number of cores that are used}

\item{cache}{either logical or path to a directory where scores are cached.
The scores of each
motif are stored in a
separate file that contains a hash table with RefSeq identifiers and
sequence type
qualifiers as keys and the number of putative binding sites as values.
If \code{cache} is \code{FALSE}, scores will not be cached.}
}
\value{
A list with three entries:

(1) df: a data frame with the following columns:
\tabular{rl}{
  \code{motif_id} \tab the motif identifier that is used in the original
  motif library\cr
  \code{motif_rbps} \tab the gene symbol of the RNA-binding protein(s)\cr
  \code{absolute_hits} \tab the absolute frequency of putative binding
  sites per motif in all
  transcripts \cr
  \code{relative_hits} \tab  the relative, i.e., absolute divided by total,
  frequency of
  binding sites per motif in all transcripts \cr
  \code{total_sites} \tab the total number of potential binding sites \cr
  \code{one_hit}, \code{two_hits}, ... \tab number of transcripts with one,
  two,
  three, ... putative binding sites
}
(2) total_sites: a numeric vector with the total number of potential
binding sites
per transcript

(3) absolute_hits: a numeric vector with the absolute (not relative)
number of putative
binding sites per transcript
}
\description{
This function is used to count the binding sites in a set of sequences for
all or a
subset of RNA-binding protein sequence
motifs and returns the result in a data frame, which is subsequently used by
\code{\link{calculate_motif_enrichment}} to
obtain binding site enrichment scores.
}
\examples{
foreground_set <- c(
  "CAACAGCCUUAAUU", "CAGUCAAGACUCC", "CUUUGGGGAAU",
  "UCAUUUUAUUAAA", "AAUUGGUGUCUGGAUACUUCCCUGUACAU",
  "AUCAAAUUA", "AGAU", "GACACUUAAAGAUCCU",
  "UAGCAUUAACUUAAUG", "AUGGA", "GAAGAGUGCUCA",
  "AUAGAC", "AGUUC", "CCAGUAA"
)
# names are used as keys in the hash table (cached version only)
# ideally sequence identifiers (e.g., RefSeq ids) and region labels
# (e.g., 3UTR for 3'-UTR)
names(foreground_set) <- c(
  "NM_1_DUMMY|3UTR", "NM_2_DUMMY|3UTR", "NM_3_DUMMY|3UTR",
  "NM_4_DUMMY|3UTR", "NM_5_DUMMY|3UTR", "NM_6_DUMMY|3UTR",
  "NM_7_DUMMY|3UTR", "NM_8_DUMMY|3UTR", "NM_9_DUMMY|3UTR",
  "NM_10_DUMMY|3UTR", "NM_11_DUMMY|3UTR", "NM_12_DUMMY|3UTR",
  "NM_13_DUMMY|3UTR", "NM_14_DUMMY|3UTR"
)

# specific motifs, uncached
motifs <- get_motif_by_rbp("ELAVL1")
scores <- score_transcripts(foreground_set, motifs = motifs, cache = FALSE)
\dontrun{
# all Transite motifs, cached (writes scores to disk)
scores <- score_transcripts(foreground_set)

# all Transite motifs, uncached
scores <- score_transcripts(foreground_set, cache = FALSE)

foreground_df <- transite:::ge$foreground1_df
foreground_set <- foreground_df$seq
names(foreground_set) <- paste0(foreground_df$refseq, "|",
   foreground_df$seq_type)
scores <- score_transcripts(foreground_set)
}
}
\seealso{
Other matrix functions: 
\code{\link{calculate_motif_enrichment}()},
\code{\link{run_matrix_spma}()},
\code{\link{run_matrix_tsma}()},
\code{\link{score_transcripts_single_motif}()}
}
\concept{matrix functions}
