% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{run_matrix_tsma}
\alias{run_matrix_tsma}
\title{Matrix-based Transcript Set Motif Analysis}
\usage{
run_matrix_tsma(
  foreground_sets,
  background_set,
  motifs = NULL,
  max_hits = 5,
  threshold_method = "p_value",
  threshold_value = 0.25^6,
  max_fg_permutations = 1e+06,
  min_fg_permutations = 1000,
  e = 5,
  p_adjust_method = "BH",
  n_cores = 1,
  cache = paste0(tempdir(), "/sc/")
)
}
\arguments{
\item{foreground_sets}{a list of named character vectors of
foreground sequences
(only containing upper case characters A, C, G, T), where the
names are RefSeq identifiers
and sequence
type qualifiers (\code{"3UTR"}, \code{"5UTR"}, \code{"mRNA"}), e.g.
\code{"NM_010356|3UTR"}. Names are only used to cache results.}

\item{background_set}{a named character vector of background
sequences (naming follows same
rules as foreground set sequences)}

\item{motifs}{a list of motifs that is used to score the specified sequences.
If \code{is.null(motifs)} then all Transite motifs are used.}

\item{max_hits}{maximum number of putative binding sites per mRNA
that are counted}

\item{threshold_method}{either \code{"p_value"} (default) or
\code{"relative"}.
If \code{threshold_method} equals \code{"p_value"}, the default
\code{threshold_value}
 is \code{0.25^6}, which is
lowest p-value that can be achieved by hexamer motifs, the shortest
supported motifs.
If \code{threshold_method} equals \code{"relative"}, the default
\code{threshold_value}
is \code{0.9}, which is 90\% of the maximum PWM score.}

\item{threshold_value}{semantics of the \code{threshold_value} depend on
\code{threshold_method} (default is 0.25^6)}

\item{max_fg_permutations}{maximum number of foreground permutations
performed in
Monte Carlo test for enrichment score}

\item{min_fg_permutations}{minimum number of foreground permutations
performed in
Monte Carlo test for enrichment score}

\item{e}{integer-valued stop criterion for enrichment score Monte Carlo
test: aborting
permutation process after
observing \code{e} random enrichment values with more extreme values than
the actual
enrichment value}

\item{p_adjust_method}{adjustment of p-values from Monte Carlo tests to
avoid alpha error
 accumulation, see \code{\link[stats]{p.adjust}}}

\item{n_cores}{the number of cores that are used}

\item{cache}{either logical or path to a directory where scores are cached.
The scores of each
motif are stored in a
separate file that contains a hash table with RefSeq identifiers and
sequence type
qualifiers as keys and the number of putative binding sites as values.
If \code{cache} is \code{FALSE}, scores will not be cached.}
}
\value{
A list with the following components:
\tabular{rl}{
  \code{foreground_scores} \tab the result of \code{\link{score_transcripts}}
  for the foreground
  sets\cr
  \code{background_scores} \tab the result of \code{\link{score_transcripts}}
  for the background
  set\cr
  \code{enrichment_dfs} \tab a list of data frames, returned by
  \code{\link{calculate_motif_enrichment}}
}
}
\description{
Calculates motif enrichment in foreground sets versus a background
set using position
weight matrices to identify putative binding sites
}
\details{
Motif transcript set analysis can be used to identify RNA binding proteins,
whose targets are
significantly overrepresented or underrepresented in certain sets of
transcripts.

The aim of Transcript Set Motif Analysis (TSMA) is to identify the
overrepresentation
and underrepresentation of potential RBP targets (binding sites)
in a set (or sets) of
sequences, i.e., the foreground set, relative to the entire population
of sequences.
The latter is called background set, which can be composed of all
sequences of the genes
of a microarray platform or all sequences of an organism or any
other meaningful
superset of the foreground sets.

The matrix-based approach skips the \emph{k}-merization step of
the \emph{k}-mer-based approach
and instead scores the transcript sequence as a whole with a
position specific scoring matrix.

For each sequence in foreground and background sets and each
sequence motif,
the scoring algorithm evaluates the score for each sequence position.
Positions with
a relative score greater than a certain threshold are considered hits, i.e.,
putative binding sites.

By scoring all sequences in foreground and background sets, a hit count
for each motif and
each set is obtained, which is used to calculate enrichment values and
associated p-values
in the same way in which motif-compatible hexamer enrichment values are
calculated in
the k -mer-based approach. P-values are adjusted with one of the available
adjustment methods.

An advantage of the matrix-based approach is the possibility of detecting
clusters of
binding sites. This can be done by counting regions with many hits using
positional
hit information or by simply applying a hit count threshold per sequence,
e.g., only
sequences with more than some number of hits are considered. Homotypic
clusters of RBP
binding sites may play a similar role as clusters of transcription factors.
}
\examples{
# define simple sequence sets for foreground and background
foreground_set1 <- c(
  "CAACAGCCUUAAUU", "CAGUCAAGACUCC", "CUUUGGGGAAU",
  "UCAUUUUAUUAAA", "AAUUGGUGUCUGGAUACUUCCCUGUACAU",
  "AUCAAAUUA", "AGAU", "GACACUUAAAGAUCCU",
  "UAGCAUUAACUUAAUG", "AUGGA", "GAAGAGUGCUCA",
  "AUAGAC", "AGUUC", "CCAGUAA"
)
names(foreground_set1) <- c(
  "NM_1_DUMMY|3UTR", "NM_2_DUMMY|3UTR", "NM_3_DUMMY|3UTR",
  "NM_4_DUMMY|3UTR", "NM_5_DUMMY|3UTR", "NM_6_DUMMY|3UTR",
  "NM_7_DUMMY|3UTR",
  "NM_8_DUMMY|3UTR", "NM_9_DUMMY|3UTR", "NM_10_DUMMY|3UTR",
  "NM_11_DUMMY|3UTR",
  "NM_12_DUMMY|3UTR", "NM_13_DUMMY|3UTR", "NM_14_DUMMY|3UTR"
)

foreground_set2 <- c("UUAUUUA", "AUCCUUUACA", "UUUUUUU", "UUUCAUCAUU")
names(foreground_set2) <- c(
  "NM_15_DUMMY|3UTR", "NM_16_DUMMY|3UTR", "NM_17_DUMMY|3UTR",
  "NM_18_DUMMY|3UTR"
)

foreground_sets <- list(foreground_set1, foreground_set2)

background_set <- c(
  "CAACAGCCUUAAUU", "CAGUCAAGACUCC", "CUUUGGGGAAU",
  "UCAUUUUAUUAAA", "AAUUGGUGUCUGGAUACUUCCCUGUACAU",
  "AUCAAAUUA", "AGAU", "GACACUUAAAGAUCCU",
  "UAGCAUUAACUUAAUG", "AUGGA", "GAAGAGUGCUCA",
  "AUAGAC", "AGUUC", "CCAGUAA",
  "UUAUUUA", "AUCCUUUACA", "UUUUUUU", "UUUCAUCAUU",
  "CCACACAC", "CUCAUUGGAG", "ACUUUGGGACA", "CAGGUCAGCA"
)
names(background_set) <- c(
  "NM_1_DUMMY|3UTR", "NM_2_DUMMY|3UTR", "NM_3_DUMMY|3UTR",
  "NM_4_DUMMY|3UTR", "NM_5_DUMMY|3UTR", "NM_6_DUMMY|3UTR",
  "NM_7_DUMMY|3UTR",
  "NM_8_DUMMY|3UTR", "NM_9_DUMMY|3UTR", "NM_10_DUMMY|3UTR",
  "NM_11_DUMMY|3UTR",
  "NM_12_DUMMY|3UTR", "NM_13_DUMMY|3UTR", "NM_14_DUMMY|3UTR",
  "NM_15_DUMMY|3UTR",
  "NM_16_DUMMY|3UTR", "NM_17_DUMMY|3UTR", "NM_18_DUMMY|3UTR",
  "NM_19_DUMMY|3UTR",
  "NM_20_DUMMY|3UTR", "NM_21_DUMMY|3UTR", "NM_22_DUMMY|3UTR"
)

# run cached version of TSMA with all Transite motifs (recommended):
# results <- run_matrix_tsma(foreground_sets, background_set)

# run uncached version with one motif:
motif_db <- get_motif_by_id("M178_0.6")
results <- run_matrix_tsma(foreground_sets, background_set, motifs = motif_db,
cache = FALSE)

\dontrun{
# define example sequence sets for foreground and background
foreground1_df <- transite:::ge$foreground1_df
foreground_set1 <- gsub("T", "U", foreground1_df$seq)
names(foreground_set1) <- paste0(foreground1_df$refseq, "|",
  foreground1_df$seq_type)

foreground2_df <- transite:::ge$foreground2_df
foreground_set2 <- gsub("T", "U", foreground2_df$seq)
names(foreground_set2) <- paste0(foreground2_df$refseq, "|",
  foreground2_df$seq_type)

foreground_sets <- list(foreground_set1, foreground_set2)

background_df <- transite:::ge$background_df
background_set <- gsub("T", "U", background_df$seq)
names(background_set) <- paste0(background_df$refseq, "|",
  background_df$seq_type)

# run cached version of TSMA with all Transite motifs (recommended)
results <- run_matrix_tsma(foreground_sets, background_set)

# run uncached version of TSMA with all Transite motifs
results <- run_matrix_tsma(foreground_sets, background_set, cache = FALSE)

# run TSMA with a subset of Transite motifs
results <- run_matrix_tsma(foreground_sets, background_set,
  motifs = get_motif_by_rbp("ELAVL1"))

# run TSMA with user-defined motif
toy_motif <- create_matrix_motif(
  "toy_motif", "example RBP", toy_motif_matrix,
  "example type", "example species", "user"
)
results <- run_matrix_tsma(foreground_sets, background_set,
  motifs = list(toy_motif))
}

}
\seealso{
Other TSMA functions: 
\code{\link{draw_volcano_plot}()},
\code{\link{run_kmer_tsma}()}

Other matrix functions: 
\code{\link{calculate_motif_enrichment}()},
\code{\link{run_matrix_spma}()},
\code{\link{score_transcripts}()},
\code{\link{score_transcripts_single_motif}()}
}
\concept{TSMA functions}
\concept{matrix functions}
